// $Id: ALPSdata.cc,v 1.9 2009/07/29 19:36:50 qnet Exp $
//
// Copyright (c) 2009 Taylor Carr <taylor.carr@gordon.edu>
// Gordon College, Department of Mathematics and Computer Science
//
// Date started:  June 12, 2009
//
// Standardized data model storage and retrieval class for ALP programs.
// The problem description, calculated parameter solutions, and any user-
// defined functions can be written to disk using the writeALPFile method
// and providing a file name (.alps extention).  An object can be constructed
// several different ways, including populating the three pieces of data
// mentioned above, by providing an existing .alps file.

#include <iostream>
#include <fstream>
#include <sstream>
#include <iomanip>
#include "ALPSdata.h"

using namespace std;

//-----------------------------------------------------------------------------

//Create new ALPSdata object providing no parameters.  This
//constructor would be used if the user wants to manually set each
//data instance variable rather than autopopulating with previous
//data.  This constructor will likely be used very little.
ALPSdata::ALPSdata() {}

//-----------------------------------------------------------------------------

//Create new ALPSdata object from an existing alps file, populating
//the instance variables with the information in the file.
ALPSdata::ALPSdata(const string alpsFile)
{
    string line; //Variable to hold each line as it is read.
    
    //Open an input file stream to read each line of data.
    ifstream initialALPS( alpsFile.data() );
    if( initialALPS.is_open() )
    {
        getline(initialALPS, line);
        while(!initialALPS.eof())
        {
            //If we see an input data tag...
            if(line.rfind(INPUT_START,INPUT_START.length()) != string::npos)
            {
                //Loop through adding all the input data to the input vector
                //until the end of input data tag is reached.
                getline(initialALPS, line);
                while(line.rfind(INPUT_END,INPUT_END.length()) == string::npos)
                {
                    _input.push_back(line);
                    getline(initialALPS, line);
                }
            }
            //Else if we see a parameter data tag...
            else if(line.rfind(ALP_START,ALP_START.length()) != string::npos)
            {
                //Loop through adding all the parameter data to the paramter
                //vector until the end of parameter data tag is reached.
                getline(initialALPS, line);
                while(line.rfind(ALP_END,ALP_END.length()) == string::npos)
                {
                    _alpSolution.push_back(line);
                    getline(initialALPS, line);
                }
            }
            //Else if we see a user function data tag...
            else if(line.rfind(BASIS_START,BASIS_START.length()) !=
		    string::npos)
            {
                //Loop through adding all the user function data to the function
                //vector until the end of function data tag is reached.
                getline(initialALPS, line);
                while(line.rfind(BASIS_END,BASIS_END.length()) == string::npos)
                {
                    _basisInfo.push_back(line);
                    getline(initialALPS, line);
                }
            }
            //If not currently inside a set of tags and no tag is found,
            //read a line (burn off lines until a tag is reached).
            else
            {
                getline(initialALPS, line);
            }   
        } 
    }
    
    //All is entered, close the file.
    initialALPS.close();
    
}

//-----------------------------------------------------------------------------

//Create an ALPSdata object, setting the input and param data to the
//provided vectors.
ALPSdata::ALPSdata(vector<string> input, vector<string> alpSolution)
{
    _input = input;
    _alpSolution = alpSolution;
}

//-----------------------------------------------------------------------------

//Create an ALPSdata object, setting the input to the provided vector and
//building the correct param vector from supplied ALP solution information.
ALPSdata::ALPSdata( string title, vector<string> input, int n, double var[],
                    vector<string> name, vector<string> desc, double obj )
{
    _input = input;
    _alpSolution = paramsToVector( title, n, var, name, desc, obj );
}

//-----------------------------------------------------------------------------

//Create an ALPSdata object, setting the input and function source to the
//provided vectors and building the correct param vector from supplied ALP
//solution information.
ALPSdata::ALPSdata( string title, vector<string> input, int n, double var[],
                    vector<string> name, vector<string> desc, double obj,
                    vector<string> basisInfo )
{
    _input = input;
    _alpSolution = paramsToVector( title, n, var, name, desc, obj );
    _basisInfo = basisInfo;
}

//-----------------------------------------------------------------------------

//Create an ALPSdata object, setting the input to the provided vector,
//building the correct param vector from supplied ALP solution information,
//and creating the function source vector from the named file.
ALPSdata::ALPSdata( string title, vector<string> input, int n, double var[],
		    vector<string> name, vector<string> desc, double obj,
		    string functionFileName )
{
    _input = input;
    _alpSolution = paramsToVector( title, n, var, name, desc, obj );
    _basisInfo = fileToVector( functionFileName );
}

//-----------------------------------------------------------------------------

//Create an ALPSdata object, setting input, param, and user defined
//functions to their respective instance variables.
ALPSdata::ALPSdata(vector<string> input, vector<string> alpSolution,
                                                 vector<string> basisInfo)
{
    _input = input;
    _alpSolution = alpSolution;
    _basisInfo = basisInfo;
}

//-----------------------------------------------------------------------------

//Create an ALPSdata object, setting input and param data, and providing
//a file containing user defined functions that will be read.
ALPSdata::ALPSdata(vector<string> input, vector<string> alpSolution,
                                                    string functionFileName)
{
    _input = input;
    _alpSolution = alpSolution;
    _basisInfo = fileToVector(functionFileName);
}

//-----------------------------------------------------------------------------

//Create a new ALPSdata object from an existing one.
ALPSdata::ALPSdata(const ALPSdata& x)
{
    _input = x._input;
    _alpSolution = x._alpSolution;
    _basisInfo = x._basisInfo;
}

//-----------------------------------------------------------------------------

//Creates the .alps data file, with the provided name, which includes
//the input, param, and user function data.
void ALPSdata::writeALPSFile(string alpsFile)
{
    //cout << "Writing ALPS File..." << endl;    
    
    //Concatenate all information to be written to disk.
    string totalOutput = INPUT_START+"\n"+vectorToString(_input)+INPUT_END
             +"\n\n"+ALP_START+"\n"+vectorToString(_alpSolution)+ALP_END
             +"\n\n"+BASIS_START+"\n"+vectorToString(_basisInfo)+BASIS_END
             +"\n";
    
    int size = totalOutput.length();
    char buffer[size];
    
    //Enter info into a char array buffer.
    for(int i = 0; i < size; i++)
    {
        buffer[i] = totalOutput[i];
    }
    
    //Create output file stream.
    ofstream fout(alpsFile.data());
    
    //Write buffer to disk.
    fout.write(buffer, size);
    
    //Close the buffer.
    fout.close();
    
    //cout << "File written successfully." << endl;
}

//-----------------------------------------------------------------------------

//Creates a file with just the user defined functions so it may
//be used/compiled as a shared object.
void ALPSdata::writeBasisInfoFile(string fileName)
{
    //cout << "Writing user-defined function source file..." << endl;
    
    //Concatenate the vector of strings into one large string to be written.
    string functionFile = vectorToString(_basisInfo);
    
    int size = functionFile.length();
    char buffer[size];
    
    //Enter info into a char array buffer.
    for(int i = 0; i < size; i++)
    {
        buffer[i] = functionFile[i];
    }
    
    //Create output file stream.
    ofstream fout(fileName.data());
    
    //Write buffer to disk.
    fout.write(buffer, size);
    
    //Close the buffer.
    fout.close();
    
    //cout << "File written successfully." << endl;
}

//-----------------------------------------------------------------------------

//Sets the input data, converting the provided file to a vector of strings.
void ALPSdata::setInput(const string inputFile)
{
    _input = fileToVector(inputFile);
}

//-----------------------------------------------------------------------------

//Sets the input data to the provided vector.
void ALPSdata::setInput(const vector<string> input)
{
    _input = input;
}

//-----------------------------------------------------------------------------

//Sets the parameter data, converting the provided file to a vector of strings.
void ALPSdata::setALPSolution(const string alpSolution)
{
    _alpSolution = fileToVector(alpSolution);
}

//-----------------------------------------------------------------------------

//Sets the parameter data to the provided vector.
void ALPSdata::setALPSolution(const vector<string> alpSolution)
{
    _alpSolution = alpSolution;
}

//-----------------------------------------------------------------------------

//Sets the function data, converting the provided file to a vector of strings.
void ALPSdata::setBasisInfo(const string basisInfo)
{
    _basisInfo = fileToVector(basisInfo);
}

//-----------------------------------------------------------------------------

//Sets the function data to the provided vector.
void ALPSdata::setBasisInfo(const vector<string> basisInfo)
{
    _basisInfo = basisInfo;
}

//-----------------------------------------------------------------------------

//Returns the vector of the input data.
vector<string> ALPSdata::getInput()
{
    return _input;
}

//-----------------------------------------------------------------------------

//Returns the vector of the parameter data.
vector<string> ALPSdata::getALPSolution()
{
    return _alpSolution;
}

//-----------------------------------------------------------------------------

//Returns the vector of the user defined functions.
vector<string> ALPSdata::getBasisInfo()
{
    return _basisInfo;
}

//-----------------------------------------------------------------------------

//Utility method for writing the data, which are stored in vectors,
//out to disk by first converting them to a string.
string ALPSdata::vectorToString(vector<string> vts)
{
    //String to return.
    string returnStr = "";
    
    vector<string>::iterator strIter;  //Vector iterator.

    //For each string in the vector, add it to the string being returned
    //followed by a newline character.
    for(strIter = vts.begin(); strIter != vts.end(); strIter++)
    {
        returnStr += *strIter + "\n";
    }
    
    return returnStr;
}

//-----------------------------------------------------------------------------

//Utility method that reads in a file, pushing the line onto a
//vector of strings and returning the vector.
vector<string> ALPSdata::fileToVector(const string file)
{
    string line; //Each line that is to be read
    vector<string> allLines; //Vector of lines
    
    //Open an input file stream to read each line of data.
    ifstream fileToRead( file.data() );
    if( fileToRead.is_open() )
    {
        //For each line, push it onto the vector.
        getline( fileToRead, line );
        while( !fileToRead.eof() )
        {
            allLines.push_back(line);
            getline( fileToRead, line );
        }
    }

    //Close input file stream.
    fileToRead.close();
    
    return allLines;
}

//-----------------------------------------------------------------------------

//Utility method that converts ALP solution parameter data into a vector of
//strings suitable for writing to an ALPS file
vector<string> ALPSdata::paramsToVector( string title, int n, double var[],
					 vector<string> name,
					 vector<string> desc, double obj )
{
    const int VariableWidth = 20;
    vector<string> allLines;
    string s;

    allLines.push_back( "# " + title );
    allLines.push_back( "NumberOfCoefficients = " + toString( n ) );

    for ( int i = 0; i <= n; i++ )
    {
	stringstream ss;
	string s = name[i] + " = " + toString( var[i] );
	ss << left << setw( VariableWidth ) << s << "# " << desc[i];
	allLines.push_back( ss.str() );
    }

    allLines.push_back( "Objective = " + toString( obj ) );

    return allLines;
}

//-----------------------------------------------------------------------------
