// $Id: ALPSdata.h,v 1.9 2009/07/29 19:35:59 qnet Exp $
//
// Copyright (c) 2009 Taylor Carr <taylor.carr@gordon.edu>
// Gordon College, Department of Mathematics and Computer Science
//
// Date started:  June 12, 2009

#ifndef __ALPSdata_H__
#define __ALPSdata_H__

#include <sstream>
#include <string>
#include <vector>

/**
 *  Class ALPSdata
**/
class ALPSdata
{
    public:
        
        /******************************METHODS******************************/
        
        //Create new ALPSdata object providing no parameters.  This
        //constructor would be used if the user wants to manually set each
        //data instance variable rather than autopopulating with previous
        //data.  This constructor will likely be used very little.
        ALPSdata();
        
        //Create new ALPSdata object from an existing alps file, populating
        //the instance variables with the information in the file.
        ALPSdata(const std::string alpsFile);
        
        //Create an ALPSdata object, setting the input and param data to the
        //provided vectors.
        ALPSdata(std::vector<std::string> inputData, 
                           std::vector<std::string> paramData);
        
        //Create an ALPSdata object, setting the input to the provided
        //vector and building the correct param vector from supplied ALP
        //solution information.
        ALPSdata(std::string title, std::vector<std::string> inputData, int n,
                           double var[], std::vector<std::string> name,
                           std::vector<std::string> desc, double obj);

        //Create an ALPSdata object, setting the input and function source
        //to the provided vectors and building the correct param vector from
        //supplied ALP solution information.
        ALPSdata(std::string title, std::vector<std::string> inputData, int n,
                           double var[], std::vector<std::string> name,
                           std::vector<std::string> desc, double obj,
                           std::vector<std::string> functionData);

        //Create an ALPSdata object, setting the input to the provided vector,
        //building the correct param vector from supplied ALP solution
        //information, and creating the function source vector from the
        //named file.
        ALPSdata(std::string title, std::vector<std::string> inputData, int n,
                           double var[], std::vector<std::string> name,
                           std::vector<std::string> desc, double obj,
                           std::string functionFileName);
        
        //Create an ALPSdata object, setting input, param, and user defined
        //functions to their respective instance variables.
        ALPSdata(std::vector<std::string> inputData,
                           std::vector<std::string> paramData,
                           std::vector<std::string> functionData);
        
        //Create an ALPSdata object, setting input and param data, and providing
        //a file containing user defined functions that will be read.
        ALPSdata(std::vector<std::string> inputData,
                           std::vector<std::string> paramData,
                           std::string functionFileName);

        //Create a new ALPSdata object from an existing one.
	    ALPSdata(const ALPSdata& x);

        //Sets the input data via either a string as the path to a file or a
        //vector of strings that represent the lines of the file.
        void setInput(const std::string inputDataFile);
        void setInput(const std::vector<std::string> inputData);
        
        //Sets the parameter data via either a string as the path to a file
        //or a vector of strings which represent the lines of the file.
        void setALPSolution(const std::string paramDataFile);
        void setALPSolution(const std::vector<std::string> paramData);
        
        //Sets the user defined functions via either a string as the path to
        //a file or a vector of strings which represent the lines of the file.
        void setBasisInfo(const std::string functionDataFile);
        void setBasisInfo(const std::vector<std::string> functionData);
        
        //Returns the vector of the input data.
        std::vector<std::string> getInput();
        
        //Returns the vector of the ALP Solution data.
        std::vector<std::string> getALPSolution();
        
        //Returns the vector of the user defined functions.
        std::vector<std::string> getBasisInfo();
        
        //Creates the .alps data file, with the provided name, which includes
        //the input, param, and user function data.
        void writeALPSFile(std::string fileName);
        
        //Creates a file with just the user defined functions so it may
        //be used/compiled as a shared object.
        void writeBasisInfoFile(std::string fileName);        
        
        /*******************************************************************/
        
    private:

        /******************************METHODS******************************/

        //Utilty to converts ALP solution parameter data into a vector
        //of strings suitable for writing to an ALPS file
        std::vector<std::string> paramsToVector(std::string title, int n,
                                  double var[], std::vector<std::string> name,
                                  std::vector<std::string> desc, double obj);

        //Utility method that reads in a file, pushing the line onto a
        //vector of strings and returning the vector.
        std::vector<std::string> fileToVector(const std::string readFile);
        
        //Utility method for writing the data, which are stored in vectors,
        //out to disk by first converting them to a string.
        std::string vectorToString(std::vector<std::string> vts);

        //Utility to convert a generic type to a string
        // (http://notfaq.wordpress.com/2006/08/30/c-convert-int-to-string/)
        template <class T> inline std::string toString( const T& t )
            { std::stringstream ss; ss << t; return ss.str(); }

        /*************************INSTANCE VARIABLES************************/

        std::vector<std::string> _input; //The input file data
        std::vector<std::string> _alpSolution; //The parameter solution data.
        std::vector<std::string> _basisInfo; //The .so fucntion data
        
        /*******************************************************************/
};

//BEGIN and END tags for the different segments of the .alps file--------------
static const std::string INPUT_START = "<INPUT>";
static const std::string INPUT_END = "</INPUT>";
static const std::string ALP_START = "<ALP_SOLUTION>";
static const std::string ALP_END = "</ALP_SOLUTION>";
static const std::string BASIS_START = "<BASIS_INFO>";
static const std::string BASIS_END = "</BASIS_INFO>";
//-----------------------------------------------------------------------------

#endif //__ALPSdata_H__
