// $Id: Basis.h,v 1.2 2009/07/28 01:44:19 senning Exp $
//
// Copyright (c) 2009 Department of Mathematics and Computer Science
// Gordon College, 255 Grapevine Road, Wenham, MA 01984
//
// Author:  Jonathan Senning <jonathan.senning@gordon.edu>
// Written: July 23, 2009
//
// Implements a function to compute quadratic, linear, pure expoential,
// mixed exponential, rational, and indicator basis functions.

#ifndef __BASIS_H__
#define __BASIS_H__

#include <string>
#include <SPNetwork.h>

class Basis
{
private:
    template <class T> std::string toString( const T& t );
    static const double _BetaLowerBound = 0.2;
    static const double _BetaUpperBound = 0.9;

protected:
    unsigned int _mode;
    int _num;
    int _indicatorTruncation;
    int* _rationalTruncation;
    double _betaLowerBound;
    double _betaUpperBound;
    double* _beta;

public:
    static const unsigned int Quadratic = 1 << 0;
    static const unsigned int Linear    = 1 << 1;
    static const unsigned int PureExp   = 1 << 2;
    static const unsigned int MixedExp  = 1 << 3;
    static const unsigned int Rational  = 1 << 4;
    static const unsigned int Indicator = 1 << 5;

    Basis( unsigned int mode = 0 );
    Basis( SPNetwork& network, unsigned int mode = 0 );
    Basis( Basis& basis );
   ~Basis( void );
    void setNetwork( SPNetwork& network );
    int getBasisSize( int n = -1 );
    double phi( int j, int x[], int n = -1 );
    std::string phiString( int j, int n = -1 );
    void setIndicatorTruncation( int M ) { _indicatorTruncation = M; }
    void setRationalTruncation( int N );
    void setRationalTruncation( int* N );
    void setBetaLowerBound( double lb ) { _betaLowerBound = lb; }
    void setBetaUpperBound( double ub ) { _betaUpperBound = ub; }
    void setBetaBounds( double lb, double ub )
	{ _betaLowerBound = lb; _betaUpperBound = ub; }
    void setMode( unsigned int mode ) { _mode = mode; }
    void addMode( unsigned int mode ) { _mode |= mode; }
    void delMode( unsigned int mode ) { _mode &= ~mode; }
    int getMode( void ) { return _mode; }
    double* getBeta( void ) { return _beta; }
    int getIndicatorTruncation( void ) { return _indicatorTruncation; }
    int* getRationalTruncation( void ) { return _rationalTruncation; }
    double getBetaLowerBound( void ) { return _betaLowerBound; }
    double getBetaUpperBound( void ) { return _betaUpperBound; }
    bool hasFunctionType( unsigned int mode ) { return ( _mode & mode ) != 0; }
};

#endif // __BASIS_H__
