// $Id: Coefficients.cc,v 1.3 2009/07/15 15:01:42 qnet Exp $
//
// Copyright (c) 2009 Jonathan Senning <jonathan.senning@gordon.edu>
// Gordon College, Department of Mathematics and Computer Science
//
// June 10, 2009
//
// Class used to construct the coefficients for the matrix representation
// of a constraint.  Nonzero coefficients are stored consecutively and a
// index vector is used to store the actual matrix row positions.
//
// The column positions for the nonzeros should be specified using the 
// normal mathematical convention where the first index for the row is 1,
// not 0 as C++ (and CPLEX) expects.  When an instance of is class is
// created, however, it must be told whether the solver being used expects
// data indices to start with 1 (as in GLPK) or 0 (as in CPLEX).
//

#include <sstream>
#include "Coefficients.h"

//============================================================================

Coefficients::Coefficients( int offset )
// Constructor
{
    // make sure offset is nonnegative
    _offset = ( offset < 0 ? 0 : offset );

    // clear data and pad with zeros if offset is positive
    this->clear();
}

//----------------------------------------------------------------------------

void Coefficients::clear( void )
// Clears internal data and pads with zeros if offset is positive
{
    _coef.clear();
    _index.clear();
    for ( int i = 0; i < _offset; i++ )
    {
	_coef.push_back( 0.0 );
	_index.push_back( 0 );
    }
}

//----------------------------------------------------------------------------

void Coefficients::add( int i, double x )
// Adds x to the coefficient with index i unless this is the first instance
// of index i in which case x is stored as the corresponding coefficient.
{
    // do nothing if index is less than 1
    if ( i < 1 ) return;

    // search for coefficient index in list
    int col = i - 1 + _offset;
    for ( unsigned int j = 0; j < _index.size(); j++ )
    {
	if ( col == _index[j] )
	{
	    _coef[j] += x;
	    return;
	}
    }

    // new index, store coefficient
    _index.push_back( col );
    _coef.push_back( x );
}

//----------------------------------------------------------------------------

std::string Coefficients::toString( std::string sep )
{
    std::stringstream ss;
    ss << _coef[1];
    for ( unsigned int i = 2; i < _coef.size(); i++ )
    {
        ss << sep << _coef[i];
    }
    return ss.str();
}

//============================================================================
