// $Id: CommandOptions.cc,v 1.2 2009/07/05 19:48:35 senning Exp $
//
// Copyright (c) 2009 Department of Mathematics and Computer Science
// Gordon College, 255 Grapevine Road, Wenham, MA 01984
//
// Author:  Jonathan Senning <jonathan.senning@gordon.edu>
// Written: July 4, 2009
//
// Class to handle command-line options using GNU getopt_long() and allow
// help text to be set for each option.  Both short and long options are
// supported.  A method can be called to display the help text for all
// options.

#include <iostream>
#include <sstream>
#include <iomanip>
#include <vector>
#include "CommandOptions.h"

//-----------------------------------------------------------------------------

// Copy constructor
//
// Input:
//   opt         - CommandOptions object

CommandOptions::CommandOptions( const CommandOptions& opt )
{
    // create new _longOptPtrs and _longOptions vectors.  All data can be
    // copied except that we need to use the address for our _flag variable
    // rather than the one from the copied instance.

    for ( unsigned int i = 0; i < opt._longOptPtrs.size(); i++ )
    {
	_longOptPtrs.push_back( new struct option );
	_longOptPtrs.back()->name = opt._longOptPtrs[i]->name;
	_longOptPtrs.back()->val = opt._longOptPtrs[i]->val;
	_longOptPtrs.back()->has_arg = opt._longOptPtrs[i]->has_arg;
	_longOptPtrs.back()->flag =
	    ( opt._longOptPtrs[i]->flag != NULL ? &_flag : NULL );
	_longOptions.push_back( *_longOptPtrs.back() );
    }

    // okay to copy remaining vectors and data

    _description  = opt._description;
    _argument     = opt._argument;
    _shortOptions = opt._shortOptions;

    _numberOfOptions = opt._numberOfOptions;
    _flag = opt._flag;
    _done = opt._done;
}

//-----------------------------------------------------------------------------

// Destructor

CommandOptions::~CommandOptions( void )
{
    for ( unsigned int i = 0; i < _longOptPtrs.size(); i++ )
    {
	delete _longOptPtrs[i];
    }
}

//-----------------------------------------------------------------------------

// Method - handle short options
//
// Input:
//   shortName   - short option character, returned by getOption()
//   longName    - long option character string
//   desc        - option help text
//   type        - option argument type: 0=none, 1=required, 2=optional
//   argument    - option argument name
// Output:
//   nothing

void CommandOptions::setOption( const char shortName, const char* longName,
				const char* desc, const int type,
				const char* argument )
{
    // save help text and argument

    _description.push_back( desc );
    _argument.push_back( argument );

    // add option to long option structure

    _longOptPtrs.push_back( new struct option );
    _longOptPtrs.back()->name = longName;
    _longOptPtrs.back()->val = shortName;
    _longOptPtrs.back()->has_arg = ( type < 0 || type > 2 ? 0 : type );
    _longOptPtrs.back()->flag = NULL;
    _longOptions.push_back( *_longOptPtrs.back() );

    // add option to short option string

    _shortOptions.push_back( shortName );
    for ( int i = 0; i < _longOptPtrs.back()->has_arg; i++ )
    {
	_shortOptions.push_back( ':' );
    }
}

//-----------------------------------------------------------------------------

// Method - handle long-only options
//
// Input:
//   value       - option value, returned by getOption()
//   longName    - long option character string
//   desc        - option help text
//   type        - option argument type: 0=none, 1=required, 2=optional
//   argument    - option argument name
// Output:
//   nothing

void CommandOptions::setOption( const int value, const char* longName,
				const char* desc, const int type,
				const char* argument )
{
    // save help text and argument

    _description.push_back( desc );
    _argument.push_back( argument );

    // add option to long option structure

    _longOptPtrs.push_back( new struct option );
    _longOptPtrs.back()->name = longName;
    _longOptPtrs.back()->val = value;
    _longOptPtrs.back()->has_arg = ( type < 0 || type > 2 ? 0 : type );
    _longOptPtrs.back()->flag = &_flag;
    _longOptions.push_back( *_longOptPtrs.back() );
}

//-----------------------------------------------------------------------------

// Method - finish building short option string and long option structure

void CommandOptions::done( void )
{
    // save zeros into longOption structure as required by getopt_long()

    _longOptPtrs.push_back( new struct option );
    _longOptPtrs.back()->name = NULL;
    _longOptPtrs.back()->val = 0;
    _longOptPtrs.back()->has_arg = 0;
    _longOptPtrs.back()->flag = NULL;
    _longOptions.push_back( *_longOptPtrs.back() );

    _shortOptions.push_back( '\0' );    // null-terminate short option string

    _numberOfOptions = _longOptPtrs.size() - 1;

    _done = true;
}

//-----------------------------------------------------------------------------

// Method - getopt_long() wrapper; parse options from command line
//
// Input:
//   value       - option value, returned by getOption()
//   argc        - number of command line values (including program name)
//   argv        - array of pointers to command line argument character arrays
// Output:
//   c           - option value

int CommandOptions::getOption( int argc, char* argv[] )
{
    int c;

    if ( !_done )
    {
	done();
    }
    c = getopt_long( argc, argv, &_shortOptions[0], &_longOptions[0], NULL );
    return ( c == 0 ? _flag : c );
}

//----------------------------------------------------------------------------

// Method - output option help text to standard output
//
// Input:
//   width       - width of option list field
// Output:
//   nothing

void CommandOptions::showOptionsHelp( int width )
{
    using namespace std;
	
    string opts;
    for ( int i = 0; i < _numberOfOptions; i++ )
    {
	if ( _longOptPtrs[i]->flag == NULL )
	{
	    stringstream value;
	    value << (char) _longOptPtrs[i]->val;
	    opts = string( "  -" ) + value.str()
		+ string( ", --" ) + string( _longOptPtrs[i]->name );
	}
	else
	{
	    opts = string( "  --" ) + string( _longOptPtrs[i]->name );
	}

	switch ( _longOptPtrs[i]->has_arg  )
	{
	    case 1:
		opts += string( "=" ) + _argument[i];
		break;
	    case 2:
		opts += string( "[=" ) + _argument[i] + string( "]" );
		break;
	    default:
		break;
	}

	cout << left << setw( width ) << opts << _description[i] << endl;
    }
}

//-----------------------------------------------------------------------------
