// $Id: MultiIndex.cc,v 1.9 2009/10/02 01:42:00 senning Exp $
//
// Copyright (c) 2009 Jonathan Senning <jonathan.senning@gordon.edu>
// Gordon College, Department of Mathematics and Computer Science
//
// Written: February 16, 2009
// Revised: October 1, 2009
//
// Implements a multi-dimensional array index class that indexes a linear
// one-dimensional array.  The main advantage of this approach is that arrays
// of arbitrary dimension (i.e. dimension determined at run-time) can be
// handled.
//
// I first saw the key idea that inspired this class in code written by Nathan
// Walker while he was a student at Gordon College.

#include <cstdlib>
#include <string>
#include <sstream>
#include "MultiIndex.h"

//===========================================================================

void MultiIndex::setData( int ndim, int start[], int end[], int stride[] )
//
//---------------------------------------------------------------------------
// Private Method - Called by constructors to initialize object.
//
// Input:
//    int ndim         - dimension of array; should be positive integer
//    int start[]      - starting indices for each dimension
//    int end[]        - ending indices for each dimension
//    int stride[]     - stride for each dimension
//
// Output:
//    None
//---------------------------------------------------------------------------
//
{
    _outOfBounds = false;
    _ndim   = ndim;
    _extent = 1;
    _dims   = new int [_ndim];
    _start  = new int [_ndim];
    _end    = new int [_ndim];
    _stride = new int [_ndim];
    _min    = new int [_ndim];
    _max    = new int [_ndim];
    _value  = new int [_ndim];

    for ( int i = 0; i < _ndim; i++ )
    {
        _start[i]  = start[i];
        _end[i]    = end[i];
        _stride[i] = stride[i];
        _min[i]    = ( _start[i] < _end[i] ? _start[i] : _end[i] );
        _max[i]    = ( _start[i] > _end[i] ? _start[i] : _end[i] );
        _value[i]  = _start[i];
        _dims[i]   = abs( _end[i] - _start[i] ) + 1;
        _extent   *= _dims[i];
	// make sure stride moves us from start to end
        if ( ( _end[i] - _start[i] ) * _stride[i] < 0 ) _outOfBounds = true;
    }
};

//===========================================================================

MultiIndex::MultiIndex( int ndim, int start[], int end[], int stride[] )
//
//---------------------------------------------------------------------------
// Constructor - Initializes an object that iterates over each dimension of a
//    logical multidimensional array.
//
// Input:
//    int ndim         - dimension of array; should be positive integer
//    int start[]      - starting indices for each dimension
//    int end[]        - ending indices for each dimension
//    int stride[]     - stride for each dimension
//
// Output:
//    the created object
//
// Usage:
//    The following code segment creates a counter that can be used to
//    iterate through a logically defined four-dimensional array.  The first
//    index is [0,0,1,1] and the last index is [6,3,8,4].  The left-most
//    (most significant) index only takes on the values 0, 2, 4, and 6
//    because the stride is 2 for that index.
//
//        int start[]  = { 0, 0, 1, 1 };
//        int end[]    = { 6, 3, 8, 4 };
//        int stride[] = { 2, 1, 1, 1 };
//        MultiIndex Counter( 4, start, end, stride );
//---------------------------------------------------------------------------
//
{
    setData( ndim, start, end, stride );
};

//===========================================================================

MultiIndex::MultiIndex( int ndim, int start[], int end[] )
//
//---------------------------------------------------------------------------
// Constructor - Initializes an object that iterates over each dimension of a
//    logical multidimensional array.
//
// Input:
//    int ndim         - dimension of array; should be positive integer
//    int start[]      - starting indices for each dimension
//    int end[]        - ending indices for each dimension
//
// Output:
//    the created object
//
// Usage:
//    The following code segment creates a counter that can be used to
//    iterate through all positions of a logically defined four-dimensional
//    array.  The first index is [0,0,1,1] and the last index is [6,3,8,4].
//
//        int start[]  = { 0, 0, 1, 1 };
//        int end[]    = { 6, 3, 8, 4 };
//        MultiIndex Counter( 4, start, end );
//---------------------------------------------------------------------------
//
{
    int* stride = new int [ndim];
    for ( int i = 0; i < ndim; i++ )
    {
        stride[i] = 1;
    }
    setData( ndim, start, end, stride );
    delete [] stride;
};

//===========================================================================

MultiIndex::MultiIndex( int ndim, int end[] )
//
//---------------------------------------------------------------------------
// Constructor - Initializes an object that iterates over each dimension of a
//    logical multidimensional array.
//
// Input:
//    int ndim         - dimension of array; should be positive integer
//    int end[]        - ending indices for each dimension
//
// Output:
//    the created object
//
// Usage:
//    The following code segment creates a counter that can be used to
//    iterate through all positions of a logically defined four-dimensional
//    array.  The first index is [0,0,0,0] and the last index is [6,3,8,4].
//
//        int end[]    = { 6, 3, 8, 4 };
//        MultiIndex Counter( 4, end );
//---------------------------------------------------------------------------
//
{
    int* start  = new int [ndim];
    int* stride = new int [ndim];
    for ( int i = 0; i < ndim; i++ )
    {
	start[i] = 0;
        stride[i] = 1;
    }
    setData( ndim, start, end, stride );
    delete [] start;
    delete [] stride;
};

//===========================================================================

MultiIndex::MultiIndex( int ndim, int start, int end, int stride )
//
//---------------------------------------------------------------------------
// Constructor - Initializes an object that iterates over each dimension of a
//    logical multidimensional array.
//
// Input:
//    int ndim         - dimension of array; should be positive integer
//    int start        - starting index for all dimensions
//    int end          - ending index for all dimensions
//    int stride       - stride for all dimensions
//
// Output:
//    the created object
//
// Usage:
//    The following code segment creates a counter that can be used to
//    iterate through a logically defined four-dimensional array.  The first
//    index is [0,0,0,0] and the last index is [9,9,9,9].
//
//        MultiIndex Counter( 4, 0, 8, 1 );
//---------------------------------------------------------------------------
//
{
    int* startvec  = new int [ndim];
    int* endvec    = new int [ndim];
    int* stridevec = new int [ndim];
    for ( int i = 0; i < ndim; i++ )
    {
        startvec[i]  = start;
        endvec[i]    = end;
        stridevec[i] = stride;
    }
    setData( ndim, startvec, endvec, stridevec );
    delete [] startvec;
    delete [] endvec;
    delete [] stridevec;
};

//===========================================================================

MultiIndex::MultiIndex( int ndim, int start, int end )
//
//---------------------------------------------------------------------------
// Constructor - Initializes an object that iterates over each dimension of a
//    logical multidimensional array.
//
// Input:
//    int ndim         - dimension of array; should be positive integer
//    int start        - starting index for all dimensions
//    int end          - ending index for all dimensions
//
// Output:
//    the created object
//
// Usage:
//    The following code segment creates a counter that can be used to
//    iterate through all positions of a logically defined four-dimensional
//    array.  The first index is [0,0,0,0] and the last index is [9,9,9,9].
//
//        MultiIndex Counter( 4, 0, 9 );
//---------------------------------------------------------------------------
//
{
    int* startvec  = new int [ndim];
    int* endvec    = new int [ndim];
    int* stridevec = new int [ndim];
    for ( int i = 0; i < ndim; i++ )
    {
        startvec[i]  = start;
        endvec[i]    = end;
        stridevec[i] = 1;
    }
    setData( ndim, startvec, endvec, stridevec );
    delete [] startvec;
    delete [] endvec;
    delete [] stridevec;    
};

//===========================================================================

MultiIndex::MultiIndex( int ndim, int end )
//
//---------------------------------------------------------------------------
// Constructor - Initializes an object that iterates over each dimension of a
//    logical multidimensional array.
//
// Input:
//    int ndim         - dimension of array; should be positive integer
//    int end          - ending index for all dimensions
//
// Output:
//    the created object
//
// Usage:
//    The following code segment creates a counter that can be used to
//    iterate through all positions of a logically defined four-dimensional
//    array.  The first index is [0,0,0,0] and the last index is [9,9,9,9].
//
//        MultiIndex Counter( 4, 0, 9 );
//---------------------------------------------------------------------------
//
{
    int* startvec  = new int [ndim];
    int* endvec    = new int [ndim];
    int* stridevec = new int [ndim];
    for ( int i = 0; i < ndim; i++ )
    {
        startvec[i]  = 0;
        endvec[i]    = end;
        stridevec[i] = 1;
    }
    setData( ndim, startvec, endvec, stridevec );
    delete [] startvec;
    delete [] endvec;
    delete [] stridevec;    
};

//===========================================================================

MultiIndex::MultiIndex( const MultiIndex& x )
//
//---------------------------------------------------------------------------
// Copy Constructor - Creates a copy of a MultiIndex object
//
// Input:
//    const MultiIndex& x     - an exisiting MultiIndex object
//
// Output:
//    the created object
//
// Usage:
//        MultiIndex Counter2 = Counter;
//---------------------------------------------------------------------------
//
{
    _outOfBounds = x._outOfBounds;
    _ndim   = x._ndim;
    _extent = x._extent;
    _dims   = new int [_ndim];
    _start  = new int [_ndim];
    _end    = new int [_ndim];
    _stride = new int [_ndim];
    _min    = new int [_ndim];
    _max    = new int [_ndim];
    _value  = new int [_ndim];

    for ( int i = 0; i < _ndim; i++ )
    {
        _start[i]  = x._start[i];
        _end[i]    = x._end[i];
        _stride[i] = x._stride[i];
        _min[i]    = x._min[i];
        _max[i]    = x._max[i];
        _value[i]  = x._value[i];
        _dims[i]   = x._dims[i];
    }
};

//===========================================================================

MultiIndex::~MultiIndex( void )
//
//---------------------------------------------------------------------------
// Destructor - Releases memory allocated for a MultiIndex object
//---------------------------------------------------------------------------
//
{
    delete [] _dims;
    delete [] _start;
    delete [] _end;
    delete [] _stride;
    delete [] _min;
    delete [] _max;
    delete [] _value;
};

//===========================================================================

void MultiIndex::resetBounds( int start[], int end[], int stride[] )
//
//---------------------------------------------------------------------------
// Method - resets starting, ending, and optionally, stride values.
//
// Input:
//    int start[]      - starting indices for each dimension
//    int end[]        - ending indices for each dimension
//    int stride[]     - stride for each dimension (optional)
//
// Output:
//    None
//---------------------------------------------------------------------------
//
{
    _outOfBounds = false;
    for ( int i = 0; i < _ndim; i++ )
    {
	_start[i]    = start[i];
	_end[i]      = end[i];
	if ( stride != NULL )
	{
	    _stride[i]   = stride[i];
	}
	_value[i] = _start[i];
	// make sure stride moves us from start to end
	if ( ( _end[i] - _start[i] ) * _stride[i] < 0 ) _outOfBounds = true;
    }
};

//===========================================================================

void MultiIndex::resetForNext( int leastSignificantIndex )
//
//---------------------------------------------------------------------------
// Method - Adjusts the counter values so that the next time the next()
//    is called it will restore the counter to the value it had prior to
//    calling this function.  This allows next() to be used in place of
//    start().
//
// Input:
//    int leastSignificantIndex - Nonnegative value that indicates the number
//                                of least significant counter values to
//                                ignore when computing the linear index
//                                corresponding to the counter values.  If
//                                leastSignificantIndex = 0 then all counter
//                                values are used; leastSignificantIndex = 2
//                                means use all but the last (rightmost) two
//                                counter values.
//
// Output:
//    None
//
// Usage:
//        MultiIndex Counter( 4, 0, 9 );
//        ...
//        (code which increments the counter to [3,7,9,5])
//        ...
//        // reset counter; x will correspond to [0,0,0,0]
//        Counter.resetForNext();
//        int x = Counter.next();
//---------------------------------------------------------------------------
//
{
    _outOfBounds = false;
    for ( int i = 0; i < _ndim; i++ )
    {
        _value[i] = _start[i];
        if ( ( _end[i] - _start[i] ) * _stride[i] < 0 ) _outOfBounds = true;
    }
    int i = _ndim - 1 - leastSignificantIndex;
    _value[i] -= _stride[i];
}

//===========================================================================

std::string MultiIndex::toString( std::string sep )
//
//---------------------------------------------------------------------------
// Methods - Returns string corresponding to current position
//
// Input:
//    std::string sep   - (Optional, default is " ") String to use as
//                        separator between index values.
//
// Output:
//    std::string       - index string
//
// Usage:
//    Assuming Counter is a MultIndex object whose current counter is
//    [3,1,5,2], the following code will display "h[3 1 5 2]" and
//    "h[3,1,5,2]":
//
//        std::cout << "h[" << Counter.toString() << "]" << std::endl;
//        std::cout << "h[" << Counter.toString( "," ) << "]" << std::endl;
//---------------------------------------------------------------------------
//
{
    std::stringstream ss;
    ss << _value[0];
    for ( int i = 1; i < _ndim; i++ )
    {
        ss << sep << _value[i];
    }
    return ss.str();
};

//===========================================================================

