// $Id: MultiIndex.h,v 1.10 2009/10/02 01:42:50 senning Exp $
//
// Copyright (c) 2009 Jonathan Senning <jonathan.senning@gordon.edu>
// Gordon College, Department of Mathematics and Computer Science
//
// Written: February 16, 2009
// Revised: October 1, 2009 
//
// Implements a multi-dimensional array index class that indexes a linear
// one-dimensional array.  The main advantage of this approach is that arrays
// of arbitrary dimension (i.e. dimension determined at run-time) can be
// handled.
//
// I first saw the key idea that inspired this class in code written by Nathan
// Walker while he was a student at Gordon College.

#ifndef __MULTIINDEX_H__
#define __MULTIINDEX_H__

#include <string>

class MultiIndex
{
protected:
    int _ndim;          // number of dimensions this counter will handle
    int _extent;        // number of locations indexed by this counter
    int* _dims;         // array of individual dimension values
    int* _start;        // array of starting counter values
    int* _end;          // array of ending counter values
    int* _stride;       // array of stride values
    int* _min;          // array of minimal counter values
    int* _max;          // array of maximal counter values
    int* _value;        // array of counter values for each dimension
    bool _outOfBounds;  // true if at least one counter value is outside range

public:
    MultiIndex( int ndim, int start[], int end[], int stride[] );
    MultiIndex( int ndim, int start[], int end[] );
    MultiIndex( int ndim, int end[] );
    MultiIndex( int ndim, int start, int end, int stride );
    MultiIndex( int ndim, int start, int end );
    MultiIndex( int ndim, int end );
    MultiIndex( const MultiIndex& x );
   ~MultiIndex( void );
    void resetBounds( int start[], int end[], int stride[] = NULL );
    void resetForNext( int leasSignificantIndex = 0 );
    inline int start( int leastSignificantIndex = 0 );
    inline int next( int leastSignificantIndex = 0 );
    inline int prev( int leastSignificantIndex = 0 );
    inline int getOffset( int leastSignificantIndex = 0 );
    inline int getOffset( int value[] );
    inline int getDelta( int delta[] );
    bool inRange( void ) { return !_outOfBounds; };
    int* getCounterArray( int bias = 0 ) { return _value - bias; };
    int* getCounterStart( int bias = 0 ) { return _start - bias; };
    int* getCounterEnd( int bias = 0 ) { return _end - bias; };
    int* getCounterStride( int bias = 0 ) { return _stride - bias; };
    int  getCounterValue( int i ) { return _value[i]; };
    int  getCounterDim( void ) { return _ndim; };
    int  getCounterExtent( void ) { return _extent; };
    std::string toString( std::string sep = " " );

private:
    void setData( int ndim, int start[], int end[], int stride[] );
};

//----------------------------------------------------------------------------

inline int MultiIndex::getDelta( int delta[] )
//
//---------------------------------------------------------------------------
// Method - Computes the linear displacement from the current counter value
//          corresponding to the offset values in delta[].  If at any of the
//          delta values yield a counter value outside of the range of the
//          counter then zero is returned, i.e. no displacement.
//
// Input:
//    int delta[]       - array containing positive and/or negative offsets
//                        from current counter value.
// Output:
//    int               - linear displacement relative to current position
//
// Usage:
//    int d[] = { 0, 0, -1, 1 };
//    int dx = Counter.getDelta( d );
//---------------------------------------------------------------------------
//
{
    int n = 0;
    for ( int i = 0; i < _ndim; i++ )
    {
	int x = _value[i] + delta[i];
	if ( x < _min[i] || x > _max[i] ) return 0;
	n = n * _dims[i] + delta[i];
    }
    return n;
};

//----------------------------------------------------------------------------

inline int MultiIndex::start( int leastSignificantIndex )
//
//---------------------------------------------------------------------------
// Method - Resets counter values to starting values and returns index
//          corresponding to starting position.
//
// Input:
//    int leastSignificantIndex - Nonnegative value that indicates the number
//                                of least significant counter values to
//                                ignore when computing the linear index
//                                corresponding to the counter values.  If
//                                leastSignificantIndex = 0 then all counter
//                                values are used; leastSignificantIndex = 2
//                                means use all but the last (rightmost) two
//                                counter values.
//
// Output:
//    int                       - linear index of starting counter value
//
// Usage:
//    MultiIndex Counter( 4, 0, 9 );
//    ...
//    (code which increments the counter to [3,7,9,5])
//    ...
//    // reset counter; x will correspond to [0,0,0,0]
//    int x = Counter.start();
//---------------------------------------------------------------------------
//
{
    _outOfBounds = false;
    for ( int i = 0; i < _ndim; i++ )
    {
        _value[i] = _start[i];
        if ( ( _end[i] - _start[i] ) * _stride[i] < 0 ) _outOfBounds = true;
    }
    return getOffset( leastSignificantIndex );
};

//----------------------------------------------------------------------------

inline int MultiIndex::next( int leastSignificantIndex )
//
//---------------------------------------------------------------------------
// Method - Increments the current counter value by adding a stride value
//          to the least-significant value in the counter.  If the ending
//          value for the least-significant value is reached, it is reset to
//          the starting value and the next-least-significant value is
//          adjusted by the stride; this continues until an adjustment by the
//          stride does not result in a "carry" or until the most-signficant
//          counter value is adjusted past its ending value.
//
// Input:
//    int leastSignificantIndex - Nonnegative value that indicates the number
//                                of least significant counter values to
//                                ignore when computing the linear index
//                                corresponding to the counter values.  If
//                                leastSignificantIndex = 0 then all counter
//                                values are used; leastSignificantIndex = 2
//                                means use all but the last (rightmost) two
//                                counter values.
//
// Output:
//    int                       - linear index of incremented counter value
//
// Usage:
//    MultiIndex Counter( 4, 0, 9 );
//    ...
//    (code which increments the counter to [3,7,4,5])
//    ...
//    int x = Counter.next();  // x is index corresponding to [3,7,4,6]
//    int y = Counter.next( 1 ); // y corresponds to [3,7,5,0]
//    int z = Counter.next( 2 ); // z corresponds to [3,8,0,0]
//---------------------------------------------------------------------------
//
{
    int i = _ndim - 1 - leastSignificantIndex;
    _value[i] += _stride[i];
    while ( i >= 0 && _value[i] * _stride[i] > _end[i] * _stride[i] )
    {
        _value[i] = _start[i];
        if ( --i >= 0 ) _value[i] += _stride[i];
    }
    _outOfBounds = ( i < 0 );
    return getOffset( leastSignificantIndex );
};

//----------------------------------------------------------------------------

inline int MultiIndex::prev( int leastSignificantIndex )
//
//---------------------------------------------------------------------------
// Method - Decrements the current counter value by subtracting a stride value
//          from the least-significant value in the counter.  If the starting
//          value for the least-significant value is reached, it is reset to
//          the ending value and the next-least-significant value is
//          adjusted by the stride; this continues until an adjustment by the
//          stride does not result in a "carry" or until the most-signficant
//          counter value is adjusted past its starting value.
//
// Input:
//    int leastSignificantIndex - Nonnegative value that indicates the number
//                                of least significant counter values to
//                                ignore when computing the linear index
//                                corresponding to the counter values.  If
//                                leastSignificantIndex = 0 then all counter
//                                values are used; leastSignificantIndex = 2
//                                means use all but the last (rightmost) two
//                                counter values.
//
// Output:
//    int                       - linear index of incremented counter value
//
// Usage:
//    MultiIndex Counter( 4, 0, 9 );
//    ...
//    (code which increments the counter to [3,7,4,5])
//    ...
//    int x = Counter.prev();  // x is index corresponding to [3,7,4,4]
//    int y = Counter.prev( 1 ); // y corresponds to [3,7,3,0]
//    int z = Counter.prev( 2 ); // z corresponds to [3,6,0,0]
//---------------------------------------------------------------------------
//
{
    int i = _ndim - 1 - leastSignificantIndex;
    _value[i] -= _stride[i];
    while ( i >= 0 && _value[i] * _stride[i] < _start[i] * _stride[i] )
    {
        _value[i] = _end[i];
        if ( --i >= 0 ) _value[i] -= _stride[i];
    }
    _outOfBounds = ( i < 0 );
    return getOffset( leastSignificantIndex );
};

//----------------------------------------------------------------------------

inline int MultiIndex::getOffset( int leastSignificantIndex )
//
//---------------------------------------------------------------------------
// Method - Computes the linear offset from the starting counter value to
//          to the current counter value.
//
// Input:
//    int leastSignificantIndex - Nonnegative value that indicates the number
//                                of least significant counter values to
//                                ignore when computing the linear index
//                                corresponding to the counter values.  If
//                                leastSignificantIndex = 0 then all counter
//                                values are used; leastSignificantIndex = 2
//                                means use all but the last (rightmost) two
//                                counter values.
//
// Output:
//    int                       - linear index of current counter value
//
// Usage:
//    MultiIndex Counter( 4, 0, 9 );
//    ...
//    (code which increments the counter to [3,7,9,5])
//    ...
//    int x = Counter.getOffset(); // x is index corresponding to [3,7,9,5]
//    int y = Counter.getOffset( 1 ); // y corresponds to [3,7,9,0]
//    int z = Counter.getOffset( 2 ); // z corresponds to [3,7,0,0]
//---------------------------------------------------------------------------
//
{
    int n = _value[0] - _min[0];
    for ( int i = 1; i < _ndim - leastSignificantIndex; i++ )
    {
        n = n * _dims[i] + ( _value[i] - _min[i] );
    }
    for ( int i = _ndim - leastSignificantIndex; i < _ndim; i++ )
    {
        n *= _dims[i];
    }
    return n;
};

//----------------------------------------------------------------------------

inline int MultiIndex::getOffset( int value[] )
//
//---------------------------------------------------------------------------
// Method - Computes the linear offset from the starting counter value to
//          to the supplied counter value.  The offset 0 is returned if
//          the supplied counter value is invalid.
//
// Input:
//    int value[]    - Array (indexed from 0) of a valid counter value
//
// Output:
//    int            - linear index of supplied counter value
//
// Usage:
//    MultiIndex Counter( 4, 0, 9 );
//    ...
//    int a[4] = { 3, 7, 9, 5 };		      
//    ...
//    int x = Counter.getOffset( a ); // x is offset  [3,7,9,5]
//---------------------------------------------------------------------------
//
{
    if ( value[0] < _min[0] || value[0] > _max[0] ) return 0;
    int n = value[0] - _min[0];
    for ( int i = 1; i < _ndim; i++ )
    {
	if ( value[i] < _min[i] || value[i] > _max[i] ) return 0;
        n = n * _dims[i] + ( value[i] - _min[i] );
    }
    return n;
};

//----------------------------------------------------------------------------

#endif // #ifndef __MULTIINDEX_H__
