// $Id: SPNetwork.cc,v 1.25 2009/09/29 12:26:15 senning Exp $
//
// Copyright (c) 2009 Christopher Pfohl <christopher.pfohl@gordon.edu>
// Gordon College, Department of Mathematics and Computer Science
//
// June 2009
// Revised through July 2009
//
// Class used to represent a stochastic processing network.
//

#include "SPNetwork.h"
#include <iostream>
#include <sstream>
#include <iomanip>
#include <cstdlib>

using namespace std;

/******************************************************************************
** Constructor for SPNetwork.  Will read in values for topology of network.   *
** If both s() and p() values are used then the program will prefer the p()   *
** data to the s() data, it will issue a warning if it winds up using both s()*
** and p() data.                                                              *
**                                                                            *
** params:                                                                    *
**    string fileName: the file from which to read the Network data           *
**                                                                            *
** <TODO>THINK ABOUT CONTENT OF THE MESSAGES!</TODO>                          *
******************************************************************************/
SPNetwork::SPNetwork( const string filename )
{
    //Set instance variable filename
    _fileName = filename;  
    TaggedValues input( filename );
    processInput( input );
}

/******************************************************************************
** Constructor for SPNetwork.  Will read in values for topology of network.   *
** If both s() and p() values are used then the program will prefer the p()   *
** data to the s() data, it will issue a warning if it winds up using both s()*
** and p() data.                                                              *
**                                                                            *
** params:                                                                    *
**    TaggedValue object                                                      *
**                                                                            *
** <TODO>THINK ABOUT CONTENT OF THE MESSAGES!</TODO>                          *
******************************************************************************/
SPNetwork::SPNetwork( TaggedValues& input )
{
    //Set instance variable filename
    _fileName = "Not Available";  
    processInput( input );
}

/******************************************************************************
** Copy Constructor for SPNetwork.                                            *
**                                                                            *
** params:                                                                    *
**    const SPNetwork& network                                                *
**                                                                            *
** <TODO>THINK ABOUT CONTENT OF THE MESSAGES!</TODO>                          *
******************************************************************************/
SPNetwork::SPNetwork( const SPNetwork& network )
{
    // copy variables

    _fileName      = network._fileName;
    _numClasses    = network._numClasses;
    _numServers    = network._numServers;
    _serverpools   = network._serverpools;
    _stochastic    = network._stochastic;
    _probabilistic = network._probabilistic;
    _flexible      = network._flexible;
    _Lambda        = network._Lambda;

    // copy vectors

    _mu = network._mu;
    _p  = network._p;
    _feasibleActions = network._feasibleActions;

    // copy arrays

    _k = new int[_numServers + 1];

    for ( int i = 1; i <= _numServers; i++ )
    {
	_k[i] = network._k[i];
    }

    _c         = new double[_numClasses + 1];
    _sigma     = new int[_numClasses + 1];
    _lambda    = new double[_numClasses + 1];
    _nonidling = new bool[_numClasses + 1];
    if( !_probabilistic ) _s = new int[_numClasses + 1];

    for ( int i = 1; i <= _numClasses; i++ )
    {
	_c[i]         = network._c[i];
	_sigma[i]     = network._sigma[i];
	_lambda[i]    = network._lambda[i];
	_nonidling[i] = network._nonidling[i];
	if( !_probabilistic ) _s[i] = network._s[i];
    }
}

/**************
** Destructor *
***************/
SPNetwork::~SPNetwork()
{
    delete [] _lambda;
    delete [] _c;
    delete [] _sigma;
    delete [] _k;
    if( !_probabilistic ) delete [] _s;
    delete [] _nonidling;
    for ( unsigned int i = 0; i < _feasibleActions.size(); i++ )
    {
	deallocateArray( _feasibleActions[i], 2, 
			 _numServers + 1, _numClasses + 1 );
    }
}

/*****************************************************
** Initialize network parameters from input object   *
******************************************************/
void SPNetwork::processInput( TaggedValues& input )
{  
    //Get number of classes
    if( (_numClasses = input.getParamValue("classes", BAD_INT)) == BAD_INT )
    {
        cerr << "ERROR: Cannot read 'classes'" << endl;
        exit( EXIT_FAILURE );
    }
    
    //Get number of servers, relies on '&&' being short-circuit
    if( (_numServers = input.getParamValue("servers", BAD_INT)) == BAD_INT &&
        (_numServers = input.getParamValue("pools", BAD_INT)) == BAD_INT  &&
        (_numServers = input.getParamValue("serverpools", BAD_INT)) == BAD_INT
      )
    {
        cerr << "ERROR: Cannot read 'servers,' 'server pools,' or 'pools'" 
             << endl;
        exit( EXIT_FAILURE );
    }

    // Get service rates (inherent in this is the service mapping).  This
    // class REQUIRES mu to be doubly indexed.  It is NOT backwards compatible
    // with previous .ini files, although previous .ini files can be made
    // compatible by adding lines: "mu("+sigma[i]+","+i+")" for each mu(i) in
    // the original
    _mu = vector<MatrixEntry>();
    _stochastic = false;
    _serverpools = false;
    _flexible = false;
    _sigma = new int[_numClasses+1];
    _Lambda = 0.0;

    _c = new double[_numClasses + 1];
    for ( int i = 1; i <= _numClasses; i++ )
    {
        double c = input.getParamValue("c(" + toString(i) + ")", BAD_DBL );
        if ( c != BAD_DBL )
        {
            _c[i] = c;
        }
        else
        {
            cerr << "ERROR:" << endl;
            cerr << "No cost data found for class << " << i << endl;
            exit( EXIT_FAILURE );
        }
    }

    _k = new int[_numServers + 1];
    for ( int i = 1; i <= _numServers; i++ ) //How many servers in the pool
    {
        //If the value read in for _k[i] != 1, then we are using server pools
	_k[i] = input.getParamValue( "K(" + toString(i) + ")", 1 );
        if ( _k[i] != 1 )
            {
                _serverpools = true;
                _stochastic = true;
            }
    }

    _nonidling = new bool[_numClasses + 1];
    for ( int i = 1; i <= _numClasses; i++ )
    {
        _nonidling[i] =
	    ( input.getParamValue("nonidling(" + toString(i) + ")", 0) != 0 );
    }
    
    // The following variables serve to make sure that each server is serving 
    // something and each queue is being served.
    
    bool serves[_numServers+1]; //serves[i] true if server i serving at least 1
    for ( int i = 1; i <= _numServers; i++ ) serves[i] = false; //initialize

    bool served[_numClasses+1]; //served[i] true if queue served by at least 1
    for ( int i = 1; i <= _numClasses; i++ ) served[i] = false; //initialize

    //Initialize _mu
    for ( int i = 1; i <= _numServers; i++ )
    {
        double maxMu = 0.0; //The largest mu for this server.
        for ( int j = 1; j <= _numClasses; j++ )
        {
            string mu = "mu(" + toString(i) + "," + toString(j) + ")";
            double value = input.getParamValue( mu, 0.0 );
                
            if ( value != 0.0 )
            {
		if ( served[j] )
		{
		    _stochastic = true; //Class is already served
		    _flexible = true;
		}
                serves[i] = true;
                served[j] = true;
                MatrixEntry nextEntry;
                nextEntry.i = i;
                nextEntry.j = j;
                nextEntry.val = value;
                if ( maxMu < value ) maxMu = value;
                _mu.push_back( nextEntry );
                _sigma[j] = i; //Will override old data if present!
            }
        }
        _Lambda += _k[i] * maxMu; //Add the largest mu for each server.
    }
    
    //The following two loops check the feasibility of _mu[][]
    for ( int i = 1; i <= _numServers; i++ )
    {
        if ( serves[i] == false )
        { 
            cerr << "ERROR: Server " << i << " does not serve any classes!"
                << endl;
            exit( EXIT_FAILURE );
        }
    }
    for ( int i = 1; i <= _numClasses; i++ )
    {
        if ( served[i] == false ) 
        {
            cerr << "ERROR: Class " << i << " is not served by any servers!"
                << endl;
            exit( EXIT_FAILURE );
        }
    }
    
    //Get Arrival Rates (inherent in this is the entrance routing info.)
    _lambda = new double[_numClasses+1];
    for ( int i = 1; i <= _numClasses; i++ )
    {
        _lambda[i] = input.getParamValue("lambda("+toString(i)+")", 0.0 );
        _Lambda += _lambda[i];
    }
    
    //Initialize the transition probability matrix
    _p = vector<MatrixEntry>();
    _s = new int[_numClasses+1];
    _probabilistic = false;  //When we start we don't know what kind of network
    bool deterministic = false;//we have (prob or det?)
    for ( int i = 1; i <= _numClasses; i++ ) //For each from-class
    {
        double rowValue = 0.0; //The total transition-out rate for this class
        bool exit_read = false; //True if exit for i has been read in.
        int successor = input.getParamValue("s(" + toString(i) + ")", 
            BAD_INT );
        if ( successor != BAD_INT )
        {
            deterministic = true;
            MatrixEntry nextPoint;
            nextPoint.i = i;
            nextPoint.j = successor;
            nextPoint.val = 1.0;
            _p.push_back( nextPoint );
            rowValue = 1.0;
        }
        for ( int j = 0; j <= _numClasses; j++ )
        {
            double prob = input.getParamValue("p(" + toString( i ) + "," + 
                toString( j ) + ")", BAD_DBL );
            if ( prob != BAD_DBL )
            {
                if ( j == 0 ) exit_read = true;
                _probabilistic = true;
                MatrixEntry nextPoint;
                nextPoint.i = i;
                nextPoint.j = j;
                nextPoint.val = prob;
                _p.push_back( nextPoint );
                rowValue += prob;
            }
        }
        if ( _probabilistic && deterministic )
        {
            cerr << "ERROR:" << endl;
            cerr << "Both successor [s(i)] data and transition probability "
                << "[p(i,j)] data found!" << endl;
            exit( EXIT_FAILURE );
        }
        if ( !exit_read && rowValue != 1.0 ) //If we need an exit value
        {
            MatrixEntry nextPoint;
            nextPoint.i = i;
            nextPoint.j = 0;
            nextPoint.val = 1.0 - rowValue;
            rowValue += nextPoint.val;
            _p.push_back(nextPoint);
        }
        if ( rowValue != 1.0 )
        {
            cerr << "WARNING:" << endl;
	    cerr << "Probability of transitioning out of class " << i
		 << " is not 1.0" << endl;
        }
    }

    // build successor array if deterministic routing is used
    if ( !_probabilistic )
    {
	for ( int i = 1; i <= _numClasses; i++ )
	{
	    for ( unsigned int n = 0; n < _p.size(); n++ )
	    {
		if ( _p[n].i == i )
		{
		    _s[i] = _p[n].j;
		}
	    }
	}
    }
}

/*****************************************************
** Prints the topology of the network to the screen. *
******************************************************/
void SPNetwork::printNetwork()
{
    int width = 7; //column width

    //cout << "Source File: " << _fileName << endl << endl;
    cout << "Network type: "
	 << ( _stochastic ? "Stochastic Processing" : "Multiclass Queueing" ) 
	 << endl;
    cout << "Number of classes: " << _numClasses << endl;
    cout << "Number of servers: " << _numServers;
    cout << " (pool sizes:";
    for ( int i = 1; i <= _numServers; i++ )
    {
	cout << ' ' << _k[i];
    }
    cout << ")" << endl << endl;

    cout << "class          =";
    for ( int i = 1; i <= _numClasses; i++ )
    {
	    cout << ' ' << setw( width ) << i;
    }
    cout << endl;

    cout << "--------------  ";
    for ( int i = 1; i <= _numClasses; i++ )
    {
	cout << ' ';
	for ( int j = 0; j < width; j++ ) cout << '-';
    }
    cout << endl;
    
    // lambda
    cout << "arrival rates  =";
    for ( int i = 1; i <= _numClasses; i++ )
    {
        cout << ' ' << setw( width ) << _lambda[i];
    }
    cout << endl;
    
    // c
    cout << "holding costs  =";
    for ( int i = 1; i <= _numClasses; i++ )
    {
	cout << ' ' << setw( width ) << _c[i];
    }
    cout << endl;
    
    if ( !_probabilistic )
    {
        // s
        cout << "successor class=";
        for ( int i = 1; i <= _numClasses; i++ )
        {
	    cout << ' ' << setw( width ) << _s[i];
        }
        cout << endl;
    }
    
    // sigma
    cout << "assigned server=";
    for ( int i = 1; i <= _numClasses; i++ )
    {
	string srvs;
        for ( unsigned int m = 0; m < _mu.size(); m++ )
        {
            if ( _mu[m].j == i )
            {
		srvs += toString( _mu[m].i ) + string( "," );
            }
        }
	srvs.erase( srvs.length() - 1 ); // remove trailing comma
	cout << ' ' << setw( width ) << srvs;
    }
    cout << endl;

    // nonidling (show only if needed)
    bool useNonIdling = false;
    for ( int i = 1; i <= _numClasses; i++ )
    {
	useNonIdling |= _nonidling[i];
    }
    if ( useNonIdling )
    {
	cout << "nonidling flag =";
	for ( int i = 1; i <= _numClasses; i++ )
	{
	    cout << ' ' << setw( width ) << _nonidling[i];
	}
	cout << endl;
    }

    // p(i,j)
    if ( _probabilistic )
    {
	cout << endl;
	cout << "transition probabilities:" << endl;
	cout << "       to:      ";
	for ( int i = 1; i <= _numClasses; i++ )
	{
	    cout << ' ' << setw( width ) << i;
	}
	cout << ' ' << setw( width ) << "exit" << endl;
	cout << "from:---------  ";
	for ( int i = 1; i <= _numClasses + 1; i++ )
	{
	    cout << ' ';
	    for ( int j = 0; j < width; j++ ) cout << '-';
	}
	cout << endl;
	for ( int i = 1; i <= _numClasses; i++ )
	{
	    cout << setw( 4 ) << i << " |          ";
	    for ( int j = 1; j <= _numClasses; j++ )
	    {
		bool out = false;
		for ( unsigned int n = 0; n < _p.size(); n++ )
		{
		    if ( _p[n].i == i && _p[n].j == j )
		    {
			cout << ' ' << setw( width ) << _p[n].val;
			out = true;
			break;
		    }
		}
		if ( !out )
		{
		    cout << " " << setw( width ) << 0.0;
		}
	    }
	    // check for exit probablity
	    bool out = false;
	    for ( unsigned int n = 0; n < _p.size(); n++ )
	    {
		if ( _p[n].i == i && _p[n].j == 0 )
		{
		    cout << ' ' << setw( width ) << _p[n].val << endl;
		    out = true;
		    break;
		}
	    }
	    if ( !out )
	    {
		cout << " " << setw( width ) << 0.0 << endl;
	    }
	}
    }

    // mu(i,j)
    cout << endl;
    cout << "service rates:" << endl;
    cout << "       class:   ";
    for ( int i = 1; i <= _numClasses; i++ )
    {
        cout << ' ' << setw( width ) << i;
    }
    cout << endl;
    cout << "server:-------   ";
    for ( int i = 1; i <= _numClasses; i++ )
    {
	cout << " ";
	for ( int j = 0; j < width; j++ ) cout << '-';
    }
    cout << endl;
    for ( int i = 1; i <= _numServers; i++ )
    {
        cout << setw( 4 ) << i << " |          ";
        for ( int j = 1; j <= _numClasses; j++ )
        {
            bool out = false;
            for ( unsigned int n = 0; n < _mu.size(); n++ )
            {
                if ( _mu[n].i == i && _mu[n].j == j )
                {
                    cout << ' ' << setw( width ) << _mu[n].val;
                    out = true;
                    break;
                }
            }
            if ( !out )
            {
                cout << " " << setw( width ) << 0.0;
            }
        }
        cout << endl;
    }
    cout << endl;
    
    double arrivalRate = 0.0;
    for ( int i = 1; i <= _numClasses; i++ )
    {
        arrivalRate += _lambda[i];
    }
    cout << "Total arrival rate: " << arrivalRate << endl;
    cout << "Total event rate:   " << _Lambda << endl;
}

/******************************************************************************
** Returns the array often referred to as "sigma."  sigma[i] = the server which
** services this class
** 
** precondition: The object represents a multiclass queuing network, or a
**               stochastic processing network with server pools, but only one
**               server serving any given class
** return value:
**      int*: array of server numbers.
******************************************************************************/
int* SPNetwork::getServerForClass()
{
    if( !_flexible ) //If no class is served by more than one 
    {
        return _sigma;
    }
    else
    {
        cout << "ERROR:" << endl;
        cout << "   You cannot ask for the class->server mapping for a"
             << " flexible network!" << endl;
        exit( EXIT_FAILURE );
    }
}

/******************************************************************************
** Returns the array often referred to as "s."  s[i] = the class to which all
** completions at class 
******************************************************************************/
int* SPNetwork::getSuccessors()
{
    if( !_probabilistic )
    {
        return _s;
    }
    else
    {
        cout << "ERROR:" << endl;
        cout << "   You cannot ask for the successor array for a probabilistic"
             << " network!" << endl;
        exit( EXIT_FAILURE );
    }
}

/******************************************************************************
** Accessor for the array of completion rate MatrixEntry's for this network.
** It is best to use it this way because mu is most likely sparse.
** Each MatrixEntry contains the following:
**     i: the server number
**     j: the class number
**     val: the completion rate of server i for class j
**      
** params:
**     int& size: returns the length of the MatrixEntry array
**
** return value:
**     MatrixEntry*: array of all the non-zero entries in transition
**                 probability matrix.  The array's size is returned in the
**                 size parameter.
******************************************************************************/
SPNetwork::MatrixEntry* SPNetwork::getServiceRates( int& size )
{
    size = (int) _mu.size();
    return &_mu[0];
}

/******************************************************************************
** Accessor for the array of Probability MatrixEntry's for this network.
** Each MatrixEntry contains the following:
**     i: the i index of the non-zero in the transition probability matrix
**     j: the j index of the non-zero in the transition probability matrix
**     val: the probability of transitioning from class i to class j.
**      
** params:
**     int& size: returns the length of the MatrixEntry array
**
** return value:
**     MatrixEntry*: an array of all the non-zero entries in transition 
**                 probability matrix.  The array's size is returned in the
**                 size parameter.
******************************************************************************/
SPNetwork::MatrixEntry* SPNetwork::getTransitionProbabilities( int& size )
{
    size = (int) _p.size();
    return &_p[0];
}

/******************************************************************************
** Utility method to convert a generic type to a string
** (From http://notfaq.wordpress.com/2006/08/30/c-convert-int-to-string/)
**
** params:
**     const T& t: a parameter to be converted to a string
**
** return value:
**     string: a string representing the parameter
******************************************************************************/
template <class T>
string SPNetwork::toString( const T& t )
{
    std::stringstream ss;
    ss << t;
    return ss.str();
}

/******************************************************************************
** Initialize the two-dimensional action array
**
** Input:
**   int** u                - action array
**
** Output:
**   int** u                - initialized action array 
**
** The entire u[][] array is initalized to zero then the last element is
** set to -1 so that when it is incremented it will be zero.
******************************************************************************/
void SPNetwork::initGetNextAction( int** u )
{
    for ( int i = 0; i <= _numServers; i++ )
    {
	for ( int j = 0; j <= _numClasses; j++ )
	{
	    u[i][j] = 0;
	}
    }
    u[_numServers][_numClasses] = -1;
}

/******************************************************************************
** Initialize the two-dimensional action array
**
** Input:
**   int* u                 - action array
**
** Output:
**   int* u                 - initialized action array 
**
** The entire u[] array is initalized to zero then the last element is
** set to -1 so that when it is incremented it will be zero.
******************************************************************************/
void SPNetwork::initGetNextAction( int* u )
{
    for ( int i = 0; i <= _numClasses; i++ )
    {
	u[i] = 0;
    }
    u[_numClasses] = -1;
}

/******************************************************************************
** Increment the action stored in the two-dimensional action array u[][]
** 
** Input:
**   int* u                 - action array
**   int  uBound            - value we increment each entry up to (default 1)
**
** Output:
**   int* u                 - incremented action array
**   bool                   - true if we've not yet reached final action
**
** This idea came to us from Nathan Walker
******************************************************************************/
bool SPNetwork::getNextAction( int* u, int uBound )
{
    int i = _numClasses;
    u[i]++;
    while ( i > 0 && u[i] > uBound )
    {
	u[i--] = 0;
	u[i]++;
    }
       
    return i > 0;
}

/******************************************************************************
** Increment the action stored in the two-dimensional action array u[][]
** 
** Input:
**   int* u                 - action array
**   int* uBound            - array of values we increment each entry up to
**
** Output:
**   int* u                 - incremented action array
**   bool                   - true if we've not yet reached final action
**
** This idea came to us from Nathan Walker
******************************************************************************/
bool SPNetwork::getNextAction( int* u, int* uBound )
{
    int i = _numClasses;
    u[i]++;
    while ( i > 0 && u[i] > uBound[i] )
    {
	u[i--] = 0;
	u[i]++;
    }
       
    return i > 0;
}

/******************************************************************************
** Increment the action stored in the two-dimensional action array u[][]
** 
** Input:
**   int** u                - action array
**   int** uBound           - upper bounds on each u_ij entry
**
** Output:
**   int** u                - incremented action array
**   bool                   - true if we've not yet reached final action
**
** This extension of an idea we first saw in code by Nathan Walker was
** suggested by Christopher Pfohl.
******************************************************************************/
bool SPNetwork::getNextAction( int** u, int** uBound )
{
    int i = _numServers;
    int j = _numClasses;
    u[i][j]++;
    while ( i > 0 && u[i][j] > uBound[i][j] )
    {
	u[i][j--] = 0;
	if ( j == 0 )
	{
	    i--;
	    j = _numClasses;
	}
	u[i][j]++;
    }
       
    return i > 0;
}

/******************************************************************************
** Determine if a given action is feasible based on network parameters
**
** Input:
**   int* u                 - action array
**   
** Output:
**   bool                   - true if action is feasible, false otherwise
******************************************************************************/
bool SPNetwork::isFeasible( int* u )
{
    if ( _flexible )
    {
        cout << "ERROR:" << endl;
        cout << "   isFeasible(int* u) does not work with flexible networks."
	     << endl << "         Try isFeasible(int** u)." << endl;
        exit( EXIT_FAILURE );
    }

    bool feasible = true; //Assume feasible unless proven otherwise

    // eliminate actions when the number of jobs being worked on by
    // a server pool exceeds the number of servers in the pool

    for ( int i = 1; i <= _numServers; i++ )
    {
	int sum = 0;
	for ( int j = 1; j <= _numClasses; j++ )
	{
	    if ( _sigma[j] == i )
	    {
		sum += u[j];
	    }
	}
	if ( sum > 1 )
	{
	    feasible = false;
	    break;
	}
    }
    return feasible;
}

/******************************************************************************
** Determine if a given action is feasible based on network parameters
**
** Input:
**   int** u                - action array
**   
** Output:
**   bool                   - true if action is feasible, false otherwise
******************************************************************************/
bool SPNetwork::isFeasible( int** u )
{
    int muLength = _mu.size();
    bool feasible = true; //Assume feasible unless proven otherwise

    // eliminate actions when the number of jobs being worked on by
    // a server pool exceeds the number of servers in the pool

    for ( int i = 1; i <= _numServers; i++ )
    {
	int sum = 0;
	for ( int j = 1; j <= _numClasses; j++ )
	{
	    sum += u[i][j];
	}
	if ( sum > _k[i] )
	{
	    feasible = false;
	    break;
	}
    }
    if ( feasible )
    {
	// eliminate actions where server works on class with service
	// rate of zero (i.e. server does not serve class)

	for ( int i = 1; i <= _numServers; i++ )
	{
	    for ( int j = 1; j <= _numClasses; j++ )
	    {
		bool iServesJ = false;
		for ( int m = 0; m < muLength; m++ )
		{
		    if ( _mu[m].i == i && _mu[m].j == j && _mu[m].val != 0 )
		    {
			iServesJ = true;
			break;
		    }
		}
		if ( u[i][j] != 0 && !iServesJ )
		{
		    feasible = false;
		    break;
		}
	    }
	    if ( !feasible ) break;
	}
    }

    return feasible;
}

/******************************************************************************
** Determine if a given action is feasible based on current state
**
** Input:
**   int* u                      - action array
**   int x[]                     - current state
**   
** Output:
**   bool                        - true if action is feasible, false otherwise
ERROR:   isStateFeasible(int* u, int x[]) does not work with flexible networks.
         Try isStateFeasible(int**u, int

******************************************************************************/
bool SPNetwork::isStateFeasible( int* u, int x[] )
{
    if ( _flexible )
    {
        cout << "ERROR:" << endl;
        cout << "   isStateFeasible(int* u, int x[]) does not work with"
	     << " flexible networks." << endl
	     << "         Try isStateFeasible(int** u, int x[])." << endl;
        exit( EXIT_FAILURE );
    }

    bool feasible = true; //Assume feasible unless proven otherwise

    // eliminate actions where number of servers working on a class
    // exceeds the number of jobs in that class.  If network uses
    // deterministic routing then make sure no server for a nonempty
    // final class is idling

    for ( int j = 1; j <= _numClasses; j++ )
    {
	if ( u[j] > x[j] )
	{
	    feasible = false;
	    break;
	}
	else if ( !_probabilistic && _s[j] == 0 && x[j] > 0 )
	{
	    int sum = 0;
	    for ( int i = 1; i <= _numClasses; i++ )
	    {
		if ( _sigma[i] == _sigma[j] )
		{
		    sum += u[i];
		}
	    }
	    if ( sum == 0 )
	    {
		feasible = false;
		break;
	    }
	}
    }

    return feasible;
}

/******************************************************************************
** Determine if a given action is feasible based on current state
**
** Input:
**   int** u                     - action array
**   int x[]                     - current state
**   
** Output:
**   bool                        - true if action is feasible, false otherwise
******************************************************************************/
bool SPNetwork::isStateFeasible( int** u, int x[] )
{
    bool feasible = true; //Assume feasible unless proven otherwise

    // eliminate actions where number of servers working on a class
    // exceeds the number of jobs in that class

    for ( int j = 1; j <= _numClasses; j++ )
    {
	int sum = 0;
	for ( int i = 1; i <= _numServers; i++ )
	{
	    sum += u[i][j];
	}
	if ( sum > x[j] )
	{
	    feasible = false;
	    break;
	}
    }

    return feasible;
}

/******************************************************************************
** Construct vector of network feasible actions
**
** Input:
**   nothing
**
** Output:
**   std::vector<int**>          - vector of feasible actions
******************************************************************************/
std::vector<int**> SPNetwork::getFeasibleActions( void )
{
    if ( _feasibleActions.size() > 0 )
    {
	return _feasibleActions;
    }

    int muLength = _mu.size();

    // allocate action array.  u[1..numberOfServers][1..numberOfClasses] 

    int** u = (int**) allocateArray( SZint, 2, _numServers + 1, 
				     _numClasses + 1 );

    // allocate and fill uBound array.  This array provides the upper
    // bounds for each u[i][j] entry during the search for feasible policies.

    int** uBound = (int**) allocateArray( SZint, 2, _numServers + 1,
					  _numClasses + 1 );

    for ( int i = 0; i <= _numServers; i++ )
    {
	for ( int j = 0; j <= _numClasses; j++ )
	{
	    uBound[i][j] = 0;
	}
    }
    for ( int n = 0; n < muLength; n++ )
    {
	if ( _mu[n].val > 0.0 )
	{
	    uBound[_mu[n].i][_mu[n].j] = _k[_mu[n].i];
	}
    }

    initGetNextAction( u );
    while ( getNextAction( u, uBound ) )
    {
	if ( isFeasible( u ) )
	{
	    int** a = (int**) allocateArray( SZint, 2, _numServers + 1, 
					     _numClasses + 1 );
	    for ( int i = 1; i <= _numServers; i++ )
	    {
		for ( int j = 1; j <= _numClasses; j++ )
		{
		    a[i][j] = u[i][j];
		}
	    }
	    _feasibleActions.push_back( a );
	}
    }
    
    deallocateArray( u, 2, _numServers + 1, _numClasses + 1 );
    deallocateArray( uBound, 2, _numServers + 1, _numClasses + 1 );

    return _feasibleActions;
}

/******************************************************************************
** Construct vector of network feasible actions for nonflexible networks
**
** Input:
**   nothing
**
** Output:
**   std::vector<int*>          - vector of feasible actions
******************************************************************************/
std::vector<int*> SPNetwork::getNonFlexFeasibleActions( void )
{
    if ( _feasibleActions1.size() > 0 )
    {
	return _feasibleActions1;
    }

    int u[_numServers + 1];

    initGetNextAction( u );
    while ( getNextAction( u, 1 ) )
    {
	if ( isFeasible( u ) )
	{
	    int* a = (int*) new int[_numClasses + 1];
	    for ( int i = 1; i <= _numClasses; i++ )
	    {
		a[i] = u[i];
	    }
	    _feasibleActions1.push_back( a );
	}
    }
    
    return _feasibleActions1;
}

/******************************************************************************
** Compute the number of possible routes through network
**
** Input:
**   nothing
**
** Output:
**   int                  - number of routes
******************************************************************************/
int SPNetwork::getNumberOfRoutes( void )
{
    int factor = 1;

    if ( _probabilistic )
    {
	int pLength = _p.size();
	for ( int i = 1; i <= _numClasses; i++ )
	{
	    int count = 0;
	    for ( int j = 0; j <= _numClasses; j++ ) // include exit class
	    {
		for ( int n = 0; n < pLength; n++ )
		{
		    if ( _p[n].i == i && _p[n].j == j && _p[n].val > 0.0 )
		    {
			count++;
		    }
		}
	    }
	    factor *= count;
	}
    }
    return factor;
}
