// $Id: SPNetwork.h,v 1.15 2009/07/26 18:10:43 senning Exp $
//
// Copyright (c) 2009 Christopher Pfohl <christopher.pfohl@gordon.edu>
// Gordon College, Department of Mathematics and Computer Science
//
// June 2009
//
// Class used to represent a stochastic processing network.
//

#ifndef __SPNETWORK_H__
#define __SPNETWORK_H__

#include <dynArray.h>
#include <string>
#include <vector>
#include <TaggedValues.h>

/*
* Class SPNetwork: functions to represent a general Multi-Class Queuing Network.
**/
class SPNetwork
{

public:
    //*******************************Constants*******************************//
    static const int BAD_INT = -666;
    static const double BAD_DBL = -6.66;
    //*******************************Constants*******************************//
    
    //********************************Methods********************************//
    
    //Constructors/Destructors:
    SPNetwork( const std::string fileName );
    SPNetwork( TaggedValues& input );
    SPNetwork( const SPNetwork& network );
    ~SPNetwork();
    
    //Structs:
    /* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * /
    * MatrixEntry represents a single entry in a sparce matrix.
    * Because it is used to hold both the service rates and the transition 
    * probabilities the names of i and j are kept, in order to emphasize that
    * it is a generic struct.
    * 
    * i and j represent the row and column of the entry in the sparce matrix.
    * val is the value stored at row i, column j.
    * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */
    struct MatrixEntry {
        int i, j;    // indices into sparce matrix
        double val; // val = probability of transition from class i to class j
    };
    
    //Accessor Methods:
    void printNetwork(); //Prints network Topology
    bool usesProbRouting() { return _probabilistic; }
    bool usesDetRouting()  { return !_probabilistic; }
    bool usingPools() { return _serverpools; } //returns _serverpools
    bool isStochastic() { return _stochastic; } //returns _stochastic
    int  getClasses() { return _numClasses; }  //Accessor for number of classes
    int  getServers() { return _numServers; }  //Accessor for number of servers
    double getTotalEventRate() { return _Lambda; }//Accessor: total event rate
    int  getServiceRateLength()  { return _mu.size(); } //returns length of mu
    int  getTransitionProbLength() { return _p.size();  } //returns length of p 
    int  getNumberOfRoutes();                  // returns number of routes
    
    /************************************************************************
    ** The following methods return pointers and so any changes made to the *
    ** return values in programs which utilize this class result in a       *
    ** change in the instance variables as well                             *
    ** maybe todo:
    ** In a perfect world these could be made const.  Future iterations     *
    ** might consider this, keeping in mind that it means that all users    *
    ** must remember to declare the variables into which these are being    *
    ** placed as const.
    *************************************************************************/
    double* getArrivalRates()    { return _lambda; }   // Accessor for _lambda
    double* getHoldingCosts()         { return _c; }   // Accessor for _c
    int*    getServerForClass();
    int*    getServerPoolSizes()      { return _k; } // Accessor for _k
    int*    getSuccessors();          //Fails if network defined with p's.
    bool*   getNonidling() { return _nonidling; }// Accessor for _nonidling
    MatrixEntry* getServiceRates( int& size );   // Accessor for _mu as an array
    MatrixEntry* getTransitionProbabilities( int& size ); // ...  _p as an array

    //Utility Methods:
    bool isFeasible( int** u );
    bool isStateFeasible( int** u, int x[] );
    std::vector<int**> getFeasibleActions( void );
    bool isFeasible( int* u );
    bool isStateFeasible( int* u, int x[] );
    std::vector<int*> getNonFlexFeasibleActions( void );

private:

    void processInput( TaggedValues& input ); // does the constructor's work
    template <class T> std::string toString( const T& t );
    //The next two methods are support methods for the three utility methods
    //exposed by the SPNetwork class
    void initGetNextAction( int** u );
    bool getNextAction( int** u, int** uBound );
    void initGetNextAction( int* u );
    bool getNextAction( int* u, int uBound = 1 );
    bool getNextAction( int* u, int* uBound );
    
    //********************************Methods********************************//
    
private:

    //**************************Instance Variables***************************//
    
    std::string _fileName; //The .ini filename
    bool _probabilistic;   //Whether or not this network is probabilistic
    bool _serverpools;     //Whether or not this network uses server pools
    bool _stochastic;      //Whether or not this network is stochastic (uses multiple 
                           //        servers per class OR uses server pools)
    bool _flexible;
    
    int _numClasses, _numServers; //Number of classes and number of servers
    
    //Service features:
    std::vector<MatrixEntry> _mu; //_mu[i][j]: machine i's service rate for
                                  // class j.
    
    //Routing features:
    std::vector<MatrixEntry> _p; //_p[i][j]: transition prob. from class i to j
    double* _lambda; // _lambda[i] = entrance rate for class i
    double  _Lambda; // _Lambda = total event rate.
    double* _c;      // _c[i] = standing cost for item in class i
    int* _sigma;     // _sigma[i] = the machine which serves class i. This array
                     // is meaningless if multiple machines can serve one class
    int* _s;
    int* _k;
    bool* _nonidling;// default false

    //feasible actions:
    std::vector<int*> _feasibleActions1;
    std::vector<int**> _feasibleActions;

    //**************************Instance Variables***************************//
    
};

#endif // __SPNETWORK_H__
