// $Id: SharedLibrary.cc,v 1.4 2009/08/04 14:34:51 qnet Exp $
//
// Copyright (c) 2009 Department of Mathematics and Computer Science
// Gordon College, 255 Grapevine Road, Wenham, MA 01984
//
// Author:  Jonathan Senning <jonathan.senning@gordon.edu>
// Written: 2009-06-24
//
// Class to build and dynamically link to a shared library.
// Christopher Pfohl figured out how to do this, all I did was build the
// class.

#include <iostream>
#include <fstream>
#include <cstdio>
#include <cstdlib>
#include <cstring>
#include <dlfcn.h>
#include "SharedLibrary.h"

const char* TmpDir = "/var/tmp";

//=============================================================================

SharedLibrary::SharedLibrary( const char* srcName, const char* flags )
//
//-----------------------------------------------------------------------------
// Constructor - build library and obtain handle to it.
//
// Input:
//    char* srcName     - name of file containing C++ source code for library
//
// Output:
//    the created object
//-----------------------------------------------------------------------------
//    
{
    strncpy( _srcFileName, srcName, _BufferSize );
    _tmpSrcFile = false;
    buildLibrary( flags == NULL ? "" : flags );
}

//=============================================================================

SharedLibrary::SharedLibrary( std::string srcName, std::string flags )
//
//-----------------------------------------------------------------------------
// Constructor - build library and obtain handle to it.
//
// Input:
//    char* srcName     - name of file containing C++ source code for library
//
// Output:
//    the created object
//-----------------------------------------------------------------------------
//    
{
    strncpy( _srcFileName, srcName.c_str(), _BufferSize );
    _tmpSrcFile = false;
    buildLibrary( flags.c_str() );
}

//=============================================================================

SharedLibrary::SharedLibrary( std::vector<std::string> src, std::string flags )
{
    snprintf( _srcFileName, _BufferSize, "%s/basis%06d.cc", TmpDir, getpid() );
    std::ofstream fout( _srcFileName );
    for ( unsigned int i = 0; i < src.size(); i++ )
    {
	fout << src[i] << std::endl;
    }
    fout.close();
    _tmpSrcFile = true;
    buildLibrary( flags.c_str() );
}

//=============================================================================

SharedLibrary::SharedLibrary( const SharedLibrary& shlib )
//
//-----------------------------------------------------------------------------
// Copy Constructor
//-----------------------------------------------------------------------------
//
{
    strncpy( _srcFileName, shlib._srcFileName, _BufferSize );
    strncpy( _soLibName, shlib._soLibName, _BufferSize );
    _tmpSrcFile = shlib._tmpSrcFile;
    _handle = shlib._handle;
}

//=============================================================================

void SharedLibrary::buildLibrary( const char* flags )
//
//-----------------------------------------------------------------------------
// Method - builds the library
//-----------------------------------------------------------------------------
//
{
    char libName[_BufferSize];

    snprintf( libName, _BufferSize, "qnetlib%06d", getpid() );
    snprintf( _soLibName, _BufferSize, "%s/%s.so.1.0", TmpDir, libName );

    // construct compiler-linker command to build shared library

    char cmd[_BufferSize];
    snprintf( cmd, _BufferSize,
	      "g++ %s -fPIC -shared -Wl,-soname,%s.so.1 -o %s %s",
	      flags, libName, _soLibName, _srcFileName );

    // issue the command - failure to build is fatal

    if ( system( cmd ) != 0 )
    {
	std::cerr << "Unable to compile functions" << std::endl;
	exit( EXIT_FAILURE );
    }

    // open library

    _handle = dlopen( _soLibName, RTLD_LAZY );
    if ( !_handle )
    {
	std::cerr << "Cannot open " << _soLibName << std::endl;
	std::cerr << dlerror() << std::endl;
	exit( EXIT_FAILURE );
    }
}

//=============================================================================

void* SharedLibrary::getFunction( const char* name )
//
//-----------------------------------------------------------------------------
// Method - returns pointer to function in library
//
// Input
//   char* name       - name of function in library
//
// Output
//   void*            - pointer to function.  Must be cast to desired type
//-----------------------------------------------------------------------------
//
{
    void* f = dlsym( _handle, name );
    if ( f == NULL )
    {
	std::cerr << dlerror() << std::endl;
    }
    return f;
}

//=============================================================================

void SharedLibrary::close( void )
//
//-----------------------------------------------------------------------------
// Method - close and remove the shared library
//-----------------------------------------------------------------------------
//
{
    if ( dlclose( _handle ) != 0 )
    {
	std::cerr << "Warning: unable to close " << _soLibName << std::endl;
    }
    if ( unlink( _soLibName ) != 0 )
    {
	std::cerr << "Warning: unable to remove " << _soLibName << std::endl;
    }
    if ( _tmpSrcFile && unlink( _srcFileName ) != 0 )
    {
	std::cerr << "Warning: unable to remove " << _srcFileName << std::endl;
    }
}
