// $Id: TaggedValues.cc,v 1.6 2009/07/27 13:48:53 senning Exp $
//
// Copyright (c) 2009 Taylor Carr <taylor.carr@gordon.edu>
// Gordon College, Department of Mathematics and Computer Science
//
// TaggedValues class is heavily modeled after a series of functions written
// by Nathan Walker <nathan.walker@gordon.edu> and extended and maintained
// by Jonathan Senning <jonathan.senning@gordon.edu>.
//
// Date started:  June 15, 2009
//
// For revisions on the file this class was modeled after see readTagged.cc
//
// Generic input file reader.  Will return ints and doubles originally 
// read from an ASCII input file.  To retrieve parameter values, supplying the 
// getParamValue method with the desired parameter value and a default value
// that will be returned if the parameter was not found as one of the keys in
// the class's map instance variable, _paramValues.
// 
// The main difference between TaggedValues and the file this class was
// modeled after is that this class opens the input file only once, reading in
// each line to a vector of strings.  The vector is then iterated over and the
// parameter- value pairs are inserted into a map.  Consequently whenever the
// user needs a parameter value, querying TaggedValues (with getParamValue)
// will perform a find on the map rather than having to open and iterate
// through each line of the input file for each parameter value query.
//
// Modified 2009/06/26 by Taylor Carr
// - Added copy constructor.

#include <iostream>
#include <sstream>
#include <fstream>
#include "TaggedValues.h"

using namespace std;

//-----------------------------------------------------------------------------

//Create TaggedValues object taking parameters from an input file.
TaggedValues::TaggedValues(const string inputFile)
{
    string line; //For each line to be read from the file.
    
    //Input file stream to read in the input file.
    ifstream inputFileInput(inputFile.data());
    
    //If the file opened correctly...
    if(inputFileInput.is_open())
    {
        //Add all of the lines to the virtual input file vector.
        getline(inputFileInput, line);
        while(!inputFileInput.eof())
        {
            _virtualInputFile.push_back(line);
            getline(inputFileInput, line);
        }
    }
    
    inputFileInput.close(); //Close input file stream.
    
    fillTaggedValuesMap(); //Fills the map using the vector instance var.
}

//-----------------------------------------------------------------------------

//Create TaggedValues object taking parameters from a virtual file.
TaggedValues::TaggedValues(const vector<string> virtualInputFile)
{
    _virtualInputFile = virtualInputFile; //Set param to instance var.
    fillTaggedValuesMap(); //Fills the map using the vector.
}

//-----------------------------------------------------------------------------

//Copy constructor, creates from an existing TaggedValues object.
TaggedValues::TaggedValues(const TaggedValues& copy)
{
    _virtualInputFile = copy._virtualInputFile;
    _paramValues = copy._paramValues;
}

//-----------------------------------------------------------------------------

//Removes whitespace (spaces and tabs) and comments from a string.
string TaggedValues::collapse(string str)
{
    string returnStr = "";
    
    //Remove whitespace - add any non whitespace or tab character to a string
    //that will be returned.
    string::iterator str_iter;
    for(str_iter = str.begin(); str_iter != str.end(); str_iter++)
    {
        if(!isspace(*str_iter))
        {
            returnStr += tolower( *str_iter ); //Add to return string.
        }
    }
    
    //Strip out comments - with the whitespace-less string, see if there is a
    //comment indicator '#' and set returnStr to the substring that is the
    //start of returnStr to the the comment indicator.
    unsigned long int commentPosition = returnStr.find('#', 0);
    if(commentPosition != string::npos)
    {
        returnStr = returnStr.substr(0, commentPosition);
    }
    
    return returnStr;
}

//-----------------------------------------------------------------------------

//Enters the parameter name and values as key-value pairings into a map.
void TaggedValues::fillTaggedValuesMap()
{
    vector<string>::iterator input_iter; //Iterator for _virtualInputFile vector
    string::iterator str_iter; //For iterating each line.
    string line;
    int splitVal;
    
    //For each index (line) in our virtual file:
    for(input_iter = _virtualInputFile.begin();
                    input_iter != _virtualInputFile.end(); input_iter++)
    {
        line = collapse(*input_iter); //Set equal to current line of input file.
                                      //(To retain the original input lines).

        str_iter = line.begin(); //Iterator for the line.
        splitVal = 0; //Will be the index for the assignemnt/separator char.
        
        if(line != "") //If the line isn't empty.
        {
	    string key = "";
	    string val = "";
	    str_iter = line.begin();
	    while ( *str_iter && *str_iter != '=' && *str_iter != ':' )
	    {
		key += *str_iter++;
	    }
	    if ( *str_iter ) str_iter++; // skip separator
	    while ( *str_iter )
	    {
		val += *str_iter++;
	    }
            //Insert key-value pair into the map.
            _paramValues.insert(make_pair(key, val));
        }
    }
}

//-----------------------------------------------------------------------------

//Returns the value paired with the provided key, tag.  If value isn't
//found, then the provided defaultValue will be returned.  Will return
//the value as the same type as the given defaultValue.
template <class T> 
T TaggedValues::getValue(string tag, const T& defaultValue)
{
    T value = defaultValue; //Set default value.
    
    tag = collapse(tag); //Remove whitespace from "tag" to match the map's keys.
    
    //Search for the provided tag in the map.
    map<string, string>::iterator param_iter = _paramValues.find(tag);
    
    //Find returns an iterator pointing to the value if it is found.  So if it
    //is not pointing to the end of the map (didn't find the value) then set
    //value equal to the tag/param's value.
    if(param_iter != _paramValues.end())
    {
        istringstream ins(param_iter->second);
        ins >> value;
    }
    
    //Return value, either the default set above, or the tag/param's value
    //if it was found in the map.
    return value;
}

//-----------------------------------------------------------------------------

//Called if the defaultValue is an integer.
int TaggedValues::getParamValue(string tag, int defaultValue)
{
    return getValue(tag, defaultValue);
}

//-----------------------------------------------------------------------------

//Called if the defaultValue is a double.
double TaggedValues::getParamValue(string tag, double defaultValue)
{
    return getValue(tag, defaultValue);
}

//-----------------------------------------------------------------------------

//Accessor for returning the virtual input file.
vector<string> TaggedValues::getVirtualInputFile()
{
    return _virtualInputFile;
}

//-----------------------------------------------------------------------------

//Prints all the key and value pairings in the _paramValues map.
void TaggedValues::printParamValues()
{
    string param, value;
    map<string, string>::iterator param_iter;
    
    for(param_iter = _paramValues.begin();
                    param_iter != _paramValues.end(); param_iter++)
    {
        param = param_iter->first;
        value = param_iter->second;
        cout << param << " = " << value << endl;
    }
}

//-----------------------------------------------------------------------------

//Prints the _virtualInputFile.  For testing or any other use.
void TaggedValues::printVirtualInputFile()
{
    cout << endl;
    cout << "----------------Begin virtual input file-----------------" << endl;
    
    vector<string>::iterator input_iter;
    for(input_iter = _virtualInputFile.begin(); 
                    input_iter != _virtualInputFile.end(); input_iter++)
    {
        cout << *input_iter << endl;
    }
    
    cout << "-----------------End virtual input file------------------" << endl;
    cout << endl;
}

//-----------------------------------------------------------------------------
