// $Id: dynArray.cc,v 1.5 2008/10/23 20:39:54 senning Exp $
//
// Copyright (c) 2007,2008 Jonathan Senning <jonathan.senning@gordon.edu>
// Gordon College, Department of Mathematics and Computer Science
//
// 2007-09-25
// 2008-10-23 Revised to allocate memory as single contiguous block
//
// These functions can be used to create and destroy multidimensional arrays
// whose dimensions are not known until runtime so that can be passed as
// arguments to functions without the dimensions being known at compile time.
//
// *************************************************************************
// * NOTE: the comments below describe the original versions of these      *
// *       functions and do not reflect the changes made to allow the      *
// *       entire multidimensional array to be allocated as a single       *
// *       contiguous block of memory.                                     *
// *                                                                       *
// *       They are retained here for reference.                           *
// *************************************************************************
//
// The main idea is simple; a 1D array is a sequence of consecutive data
// locations that can be referenced by offsets from the base pointer (the
// address of the first element).  A 2D array can be constructed as a 1D
// array of pointers to 1D arrays.  If this is done in C or C++ then the
// reference a[2][5] can be thought of as (a[2])[5] which means an offset of
// 5 applied to the address specified by a[2] which in turn is an offset of
// 2 from the address specified by a.  The compiler/runtime system takes care
// handing the stride of the offset which will depend on the datatype and
// size of pointers.
//
// Since a 2D array is a 1D array of 1D arrays and a 3D array is a 1D array
// of 2D arrays, etc., we can use recursion to allocate an N dimensional
// array
//
//      function allocateArray( N ) 
//      if ( N == 1 ) then
//          allocate 1D array for data
//      else
//          allocate 1D array of pointers
//          for each pointer in new array
//              allocateArray( N - 1 )
//
// Of course there are details to worry about like the actually array
// dimensions and how to keep track of the pointers, but this gives the
// basic idea.
//
// What makes this possible in C/C++ is the ability to recast pointers,
// telling the compiler how it should treat each pointer variable.  This code
// was written on a system running SuSE 10.2 Linux and using g++ (GCC) 4.1.2.
// It has been tested on RedHat Enterprise Linux WS 4 using g++ (GCC) 3.4.6
// and SGI IRIX mips 6.5 running g++ (GCC) 3.3.

#include "dynArray.h"
#include <stdarg.h>

typedef unsigned char Byte;

//----------------------------------------------------------------------------
// Private functions that support publically callable functions
//----------------------------------------------------------------------------

static void* assignPointers( Byte* q, int datasize, int ndim, int dim[] )
//
// ---------------------------------------------------------------------------
// This is a RECURSIVE function.
//
// Assigns pointers to positions of contiguous memory block pointed to by q.
//
// Parameters:
//   Byte* q          - pointer to start of contiguous memory
//   int datasize     - number of bytes required for each data entry
//   int ndim         - dimension of the array
//   int dim[ndim]    - array of ndim integers that are the array dimensions
//
// Return Value:
//   void* p          - pointer to the array; must be cast to desired type.
//                      NULL is returned if the allocation failed
// ---------------------------------------------------------------------------
//
{
    void* p = (void*) 0; // NULL

    if ( ndim == 1 )
    {
	// we've already allocated all the necessary pointers, now we
	// need to assign start address to pointer.
	p = q;
    }
    else if ( ndim > 1 )
    {
	// determine the stride for the remaining dimensions

	int bufsize = datasize;
	for ( int i = 1; i < ndim; i++ )
	{
	    bufsize *= dim[i];
	}

	// allocate a row of pointers, each of which points to an
	// (ndim-1)-dimensional array and then recursively call this
	// routine to allocate the pointers for the remaining (ndim-1)-
	// dimensions.

	ndim--;
	p = new Byte [ dim[0] * sizeof( Byte* ) ];
	for ( int i = 0; i < dim[0]; i++ )
	{
	    ((Byte**)p)[i] =
		(Byte*) assignPointers( q, datasize, ndim, &dim[1] );
	    q += bufsize;
	}
    }
    
    return p;
}

//---------------------------------------------------------------------------

static void deassignPointers( void* p, int ndim, int dim[] )
//
// --------------------------------------------------------------------------
// This is a RECURSIVE function.
//
// Releases memory previously allocated for pointers.
//
// Parameters:
//   void* p          - address of array
//   int ndim         - dimension of the array
//   int dim[ndim]    - array of n integers that are the array dimensions
//
// Return Value:
//   none
// --------------------------------------------------------------------------
//
{
    if ( ndim > 1 )
    {
	// loop through the list of pointers recursively calling self to
	// release all memory each pointer points to, then release memory
	// for this row of pointers

	ndim--;
	for ( int i = 0; i < dim[0]; i++ )
	{
	    deassignPointers( ((Byte**)p)[i], ndim, &dim[1] );
	}
	delete [] (Byte*) p;
    }
}

//----------------------------------------------------------------------------
// Public functions
//----------------------------------------------------------------------------

void* allocateArrayV( int datasize, int ndim, int dim[] )
//
// ---------------------------------------------------------------------------
// Allocates memory for an n-dimensional array with each dimension being
// supplied in the dim[] array.  While this function may be called directly,
// it is also called by allocateArray() where the individual dimensions are
// passed in a variable-length argument list.
//
// Each element in the allocated array has datasize bytes.  The array
// allocated with this function can be passed as a function parameter without
// specifying the dimensions (unlike statically allocated arrays).
//
// Parameters:
//   int datasize     - number of bytes required for each data entry
//   int ndim         - dimension of the array
//   int dim[ndim]    - array of ndim integers that are the array dimensions
//
// Return Value:
//   void* p          - pointer to the array; must be cast to desired type
//                      NULL is returned if the allocation failed
//
// Example Usage:
//
//   A two-dimensional array of integers could be allocated with
//
//     int dim[2] = { 10, 15 };
//     int** a = (int**) allocateArrayV( SZint, 2, dim );
//
//   The resulting array can be accessed as if it had been declared as
//   int a[10][15].  The constant SZint is defined in dynArray.h; it would
//   also be possible to use "sizeof( int )" in its position.
//
//   A three-dimensional array of floating point numbers with dimensions 8,
//   5, and 13 can be allocated with
//
//       int dim[3] = { 8, 5, 13 };
//       float*** b = (float***) allocateArrayV( SZfloat, 3, dim );
//
//   Elements in the array can be accessed with references like b[0][2][5].
// ---------------------------------------------------------------------------
//
{
    // determine number of bytes required for contiguous block of memory

    int bufsize = datasize;
    for ( int i = 0; i < ndim; i++ )
    {
	bufsize *= dim[i];
    }

    // allocate memory and assign pointers so it may be accessed as a
    // multidimesional array

    Byte* q = new Byte [ bufsize ];
    return assignPointers( q, datasize, ndim, dim );
}

//---------------------------------------------------------------------------

void deallocateArrayV( void* p, int ndim, int dim[] )
//
// --------------------------------------------------------------------------
// Releases memory previously allocated with allocateArrayV().
//
// Parameters:
//   void* p          - address of array
//   int ndim         - dimension of the array
//   int dim[ndim]    - array of n integers that are the array dimensions
//
// Return Value:
//   none
//
// Example Usage:
//
//   A two-dimensional array of integers previous allocated with
//
//       int dim[2] = { 10, 15 };
//       int** a = (int**) allocateArrayV( SZint, 2, dim );
//
//   can be deallocated with
//
//       deallocateArrayV( a, 2, dim );
// ---------------------------------------------------------------------------
//
{
    // find pointer to start of originally allocated contiguous block.

    Byte* q = (Byte*) p; 
    for ( int i = 1; i < ndim; i++ )
    {
	q = ((Byte**)q)[0];
    }

    // release memory for pointers and delete contiguous memory block

    deassignPointers( p, ndim, dim );
    delete [] (Byte*) q;
}

//----------------------------------------------------------------------------

void* allocateArray( int datasize, int ndim, ... )
//
// ---------------------------------------------------------------------------
// This function allocates memory for an ndim-dimensional array with each
// dimension being supplied in a variable-length argument list.  This function
// converts the argument list to an array and then calls the recursive
// function allocateArrayV() to do the actual allocation work.
//
// Each element in the allocated array has datasize bytes.  The array
// allocated with this function can be passed as a function parameter without
// specifying the dimensions (unlike statically allocated arrays).
//
// Parameters:
//   int datasize     - number of bytes required for each data entry
//   int ndim         - dimension of the array
//   ...              - ndim integer arguments separated by commas that
//                        specify the array dimensions
// Return Value:
//   void* p          - pointer to the array; must be cast to desired type
//
// Example Usage:
//
//   A two-dimensional array of integers could be allocated with
//
//       int** a = (int**) allocateArray( SZint, 2, 10, 15 );
//
//   The resulting array can be accessed as if it had been declared with
//   int a[10][15].
//
//   A three-dimensional array of floating point numbers with dimensions 8,
//   5, and 13 can be allocated with
//
//       float*** b = (float***) allocateArray( SZfloat, 3, 8, 5, 13 );
//
//   Elements in the array can be accessed with references like b[0][2][5].
// ---------------------------------------------------------------------------
//
{
    int dim[ndim];
    va_list argptr;

    // put dimension arguments into single array

    va_start( argptr, ndim );
    for ( int i = 0; i < ndim; i++ )
    {
	dim[i] = va_arg( argptr, int );
    }
    va_end( argptr );

    // create and return the requested array

    return allocateArrayV( datasize, ndim, dim );
}

//----------------------------------------------------------------------------

void deallocateArray( void* p, int ndim, ... )
//
// ---------------------------------------------------------------------------
// Releases memory previously allocated with allocateArray().
//
// Parameters:
//   void* p          - address of array
//   int ndim         - dimension of the array
//   ...              - ndim integer arguments separated by commas that
//                        specify the array dimensions
// Return Value:
//   none
//
// Example Usage:
//
//   A two-dimensional array of integers previous allocated with
//
//       int** a = (int**) allocateArray( SZint, 2, 10, 15 );
//
//   can be deallocated with
//
//       deallocateArray( a, 2, 10, 15 );
// ---------------------------------------------------------------------------
//
{
    int dim[ndim];
    va_list argptr;

    // put dimension arguments into single array

    va_start( argptr, ndim );
    for ( int i = 0; i < ndim; i++ )
    {
	dim[i] = va_arg( argptr, int );
    }
    va_end( argptr );

    // release the memory

    deallocateArrayV( p, ndim, dim );
}

//----------------------------------------------------------------------------

// End of file
