// $Id: makeArrays.cc,v 1.4 2009/07/02 01:26:01 qnet Exp $
//
// Copyright (c) 2007-2009 Department of Mathematics and Computer Science
// Gordon College, 255 Grapevine Road, Wenham, MA 01984
//
// Author:  Jonathan Senning <jonathan.senning@gordon.edu>
// Written: 2007-09-27
// Revised: 2009-06-23
//
// These functions can be used to create and destroy arrays used to hold
// policy and value data for QNET programs.  They handle the necessary
// conversion between QNET dimension conventions and C/C++ array conventions.
//
// The arrays defined here are single-dimensional but hold multidimensional
// data.  The MultiIndex class can be used to access elements in the array.
//
// It's a little confusing...  In the discussion below the word "dimension" is
// overused (actually, it's just "overloaded").

// The state space in QNET programs is the space defined by all possible
// states, i.e. values of (x1,x2,...,xn).  The values x1, x2, ..., xn are
// the number of items in the corresponding class.  Thus, in the notation
// used here, the "n" in "xn" refers to the number of classes.  In the QNET
// programs this is referred to as the "state space dimension" or "dimension
// of the state space."
//
// Since the QNET programs work with finite state spaces, there are upper-
// bounds on each of the xi values.  These bounds are called truncations
// and are usually denoted N1, N2, ..., Nn.  For each i in 1..n, we have
// that 0 <= xi <= Ni so that the N values are inclusive upper bounds.
// This means that xi can take on Ni+1 different values.  As a result, the
// array required to hold the state space values needs to have dimensions
// N1+1, N2+1, ..., Nn+1.
//
// Finally, in the case of policy data, there may be more than one policy
// decision required at each state (e.g. two different servers each of which
// can serve different classes).  If each state has only one decision then
// the dimension of the array is the same as the dimension of the state space.
// If, however, each state has more than one decision associated with it then
// the dimension of the array needs to be one more than the dimension of the
// state space. 

#include "qnet.h"

//----------------------------------------------------------------------------

int* makePolicyArray1D( int ndim, int decn, int dim[] )
{
    int size = ( decn < 1 ? 1 : decn );
    for ( int i = 0; i < ndim; i++ )
    {
	size *= ( dim[i] + 1 );
    }
    return new int [size];
}

//----------------------------------------------------------------------------

double* makeValueArray1D( int ndim, int nval, int dim[] )
{
    int size = ( nval < 1 ? 1 : nval );
    for ( int i = 0; i < ndim; i++ )
    {
	size *= ( dim[i] + 1 );
    }
    return new double [size];	   
}

//----------------------------------------------------------------------------

void destroyPolicyArray1D( void* p, int ndim, int decn, int dim[] )
{
    delete [] (int*) p;
}

//----------------------------------------------------------------------------

void destroyValueArray1D( void* p, int ndim, int nval, int dim[] )
{
    delete [] (double*) p;
}

//----------------------------------------------------------------------------

int* makePolicyArray( int ndim, int decn, int dim[] )
{
    int arrayNDim = ndim + ( decn > 1 ? 1 : 0 );
    int arrayDim[arrayNDim];
    for ( int i = 0; i < ndim; i++ )
    {
	arrayDim[i] = dim[i] + 1;
    }
    if ( decn > 1 ) arrayDim[ndim] = decn;
    return (int*) allocateArrayV( SZint, arrayNDim, arrayDim );
}

//----------------------------------------------------------------------------

double* makeValueArray( int ndim, int nval, int dim[] )
{
    int arrayNDim = ndim + ( nval > 1 ? 1 : 0 );
    int arrayDim[arrayNDim];
    for ( int i = 0; i < ndim; i++ )
    {
	arrayDim[i] = dim[i] + 1;
    }
    if ( nval > 1 ) arrayDim[ndim] = nval;
    return (double*) allocateArrayV( SZdouble, arrayNDim, arrayDim );
}

//----------------------------------------------------------------------------

void destroyPolicyArray( void* p, int ndim, int decn, int dim[] )
{
    int arrayNDim = ndim + ( decn > 1 ? 1 : 0 );
    int arrayDim[arrayNDim];
    for ( int i = 0; i < ndim; i++ )
    {
	arrayDim[i] = dim[i] + 1;
    }
    if ( decn > 1 ) arrayDim[ndim] = decn;
    deallocateArrayV( p, arrayNDim, arrayDim );
}

//----------------------------------------------------------------------------

void destroyValueArray( void* p, int ndim, int nval, int dim[] )
{
    int arrayNDim = ndim + ( nval > 1 ? 1 : 0 );
    int arrayDim[arrayNDim];
    for ( int i = 0; i < ndim; i++ )
    {
	arrayDim[i] = dim[i] + 1;
    }
    if ( nval > 1 ) arrayDim[ndim] = nval;
    deallocateArrayV( p, arrayNDim, arrayDim );
}

//----------------------------------------------------------------------------

// End of file
