// $Id: policyIO.cc,v 1.15 2009/07/02 01:27:32 qnet Exp $
//
// Copyright (c) 2007-2009 Department of Mathematics and Computer Science
// Gordon College, 255 Grapevine Road, Wenham, MA 01984
//
// Author:  Jonathan Senning <jonathan.senning@gordon.edu>
// Written: 2007-09-25
//
// These functions are used to read and write policy data files using the
// DPViewer policy file format:
//
// The first four bytes is a magic number that marks the file as a QNET
// policy file.  It should be a negative integer with the value -5.
//
// The remaining data in the file is appears as space-delimited ASCII strings
// representing integer values followed by a newline and a comment string.
// 
//  Position(s)     Name
//  -------------  ------  -------------------------------------------------  
//       1          ndim   Dimension of state space
//
//       2          decn   Number of decisions (i.e. actions stored for each
//                         state
//
//  3 .. 3+ndim-1   dim[]  State space truncations (each one less than
//                         individual state space array dimensions)
//
//  3+ndim .. K-1          Policy data stored in row-major order with all
//                         decision for a given state appearing consecutively
//
//     K .. EOF            comment string
//
// where
//     K = 3+ndim + Prod_{i=1}^ndim (dim[i]+1)
//
// ---------------------------------------------------------------------------
// Change Log
//
// 2008-06-02 Jonathan Senning <jonathan.senning@gordon.edu>
// - changed filename parameter from char* to string
//
// 2009-06-23 Jonathan Senning <jonathan.senning@gordon.edu>
// - policy data now in single dimensional array

#include <fstream>
#include <string>
#include <time.h>
#include "qnet.h"
using namespace std;

//----------------------------------------------------------------------------

static int readPolicyHeader( ifstream& fin, int maxdim, int* ndim,
			     int* decn, int dim[] )
//
// ---------------------------------------------------------------------------
// Reads a policy data file header.
//
// Parameters:
//   ifstream& fin       - file stream to read data from
//   int maxdim          - maximum dimension of the data set
//   int* ndim           - pointer to location to store state space dimension
//   int* decn           - pointer to location to store number of decisions
//   int dim[]           - array of state space dimensions.  This array must
//                         have already been declared and contain at least
//                         maxdim elements.
//
// Return Value:
//   int status          - zero if no error, negative if error
// ---------------------------------------------------------------------------
//
{
    // Read the magic number that should be stored at the start of the file.
    // If this does not match the number used for policy files then we should
    // generate a warning or an error.

    int type;
    fin.read( reinterpret_cast<char*> (&type), sizeof( int ) );
    if ( type != QNET_POLICY_MAGIC ) return QNET_BAD_MAGIC;

    // Read the data set dimension and number of decisions from the file.

    fin >> *ndim >> *decn;

    // Make sure that the caller has indicated that the dim[] array is large
    // enough to hold the dimension data we're going to put into it.

    if ( maxdim < 0 || *ndim > maxdim ) return QNET_BAD_DIMENSION;

    // Read in the individual dimensions.

    for ( int i = 0; i < *ndim; i++ )
    {
	fin >> dim[i];
    }
    
    return QNET_NORMAL;
}

//----------------------------------------------------------------------------

static void writePolicyHeader( ofstream& fout, int ndim, int decn, int dim[] )
//
// ---------------------------------------------------------------------------
// Write a policy data file header.
//
// Parameters:
//   ofstream& fout      - file stream to write data to
//   int ndim            - dimension of the data set being saved
//   int decn            - number of decisions for each state
//   int dim[ndim+1]     - array of largest indices in state space
//                          (one less than state space dimensions)
// ---------------------------------------------------------------------------
//
{
    // Write out magic number to the file.  Only policy files should have
    // this number stored in the first four bytes of the file.

    int type = QNET_POLICY_MAGIC;
    fout.write( reinterpret_cast<char*> (&type), sizeof( int ) );

    // Write state space dimension and number of decisions in each state.
    // Then write the state space dimension data.

    fout << ' ' << ndim << ' ' << decn;
    for ( int i = 0; i < ndim; i++ )
    {
	fout << ' ' << dim[i];
    }
}

//----------------------------------------------------------------------------

static void readPolicyData1D( ifstream& fin, int ndim, int dim[], int* p )
//
// ---------------------------------------------------------------------------
// Reads the data portion of a policy file.  The array to contain the
// policy data is assumed to have been created using the dynArray routines.
//
// Parameters:
//   ifstream& fin       - file stream to read data from
//   int ndim            - dimension of the array being read
//   int dim[ndim]       - array of largest indices in state space
//                          (one less than state space dimensions)
//   int* p              - pointer portion of array being read
//
// Return Value:
//   none
// ---------------------------------------------------------------------------
//
{
    int size = 1;
    for ( int i = 0; i < ndim; i++ )
    {
	size *= dim[i];
    }
    for ( int i = 0; i < size; i++ )
    {
	fin >> p[i];
    }
}

//----------------------------------------------------------------------------

static void writePolicyData1D( ofstream& fout, int ndim, int dim[], int* p )
//
// ---------------------------------------------------------------------------
// Writes the data portion of a policy file.  The array containing the
// policy data is assumed to have been created using the dynArray routines.
//
// Parameters:
//   ofstream& fout      - file stream to write data to
//   int ndim            - dimension of the array being written
//   int dim[ndim]       - array of largest indices in state space
//                          (one less than state space dimensions)
//   int* p              - pointer portion of array being written
//
// Return Value:
//   none
// ---------------------------------------------------------------------------
//
{
    int size = 1;
    for ( int i = 0; i < ndim; i++ )
    {
	size *= dim[i];
    }
    for ( int i = 0; i < size; i++ )
    {
	fout << ' ' << p[i];
    }
}

//----------------------------------------------------------------------------

static void writeComment( ofstream& fout, char* comment )
//
// ---------------------------------------------------------------------------
// Writes a comment string to a data file.  If comment is NULL then a
// string with a data-time stamp is written.
//
// Parameters:
//   ofstream& fout      - file stream to write data to
//   char* comment       - string to append to end of file
//
// Return Value:
//   none
// ---------------------------------------------------------------------------
//
{
    // Separate comment from data

    fout << endl;

    // Write comment or time stamp

    if ( comment != NULL )
    {
        fout << "COMMENT: " << comment << endl;
    }
    else
    {
	time_t t;
	time( &t );
	// '\n' is last character in ctime() string so endl is not needed
	fout << "COMMENT: " << "Policy data created at " << ctime( &t );
    }
}

//----------------------------------------------------------------------------

int getPolicyFileInfo( string path, int maxdim, int* ndim,
		       int* decn, int dim[] )
//
// ---------------------------------------------------------------------------
// Provides user access to policy file header information.
//
// Parameters:
//   string path         - name of file to read from
//   int maxdim          - maximum dimension of the data set
//   int* ndim           - pointer to location to store state space dimension
//   int* decn           - pointer to location to store number of decisions
//   int dim[]           - array of state space dimensions.  This array must
//                         have already been declared and contain at least
//                         maxdim elements.
// Return Value:
//   int status          - zero if no error, negative if error
// ---------------------------------------------------------------------------
//
{
    int status;
    ifstream fin( path.data() );
    if ( fin.is_open() )
    {
	status = readPolicyHeader( fin, maxdim, ndim, decn, dim );
    }
    else
    {
	status = QNET_BAD_FILE;
    }
    fin.close();
    return status;
}

//----------------------------------------------------------------------------

int readPolicyFile1D( string path, int ndim, int decn, int dim[], void* p )
//
// ---------------------------------------------------------------------------
// Reads a policy data file into a policy array created with the dynArray
// routines.
//
// Parameters:
//   string path         - name of file to read from
//   int ndim            - dimension of the array being read
//   int decn            - number of decision for each state
//   int dim[ndim+1]     - array of largest indices in state space
//                          (one less than state space dimensions)
//   void* p             - pointer to policy data array
//
// Return Value:
//   int status          - zero if no error, negative if error
//
// Example Usage:
//
//   To read a policy file, the dimension and decision information must
//   already be known and the array to hold the policy data must already be
//   declared.  The information necessary to do this can be obtained with
//   the getPolicyFileInfo() function.
//
//     int maxDim = 3; // largest possible dimension of data set state space
//     int ndim;
//     int decn;
//     int N[maxDim + 1];
//     getPolicyFileInfo( string("policy.u"), maxDim, &ndim, &decn, &N[1] );
//
//   Convention used in most of the QNET DP programs says that the truncation
//   values are indexed starting with 1.  We can follow this convention by
//   declaring N[] to have one extra element and passing getPolicyFileInfo()
//   the address of the N[1] element.
//
//   Let's assume that ndim = 3 and decn = 2.  Since decn > 1 we will need
//   to allocate an ndim+1 = 4 dimensional array to hold the policy data.
//   This is done with
//
//      int**** p = (int****) makePolicyArray( ndim, decn, &N[1] );
//
//   Note that the only change necessary to the above line when ndim and decn
//   have different values is the number of "*" used (i.e. the dimension of
//   the array).
//
//   Finally, we're ready to read the data:
//
//      readPolicyFile( string("policy.u"), ndim, decn, &N[1], p );
// ---------------------------------------------------------------------------
//
{
    int ndimInFile;       // Number of state space dimensions in file
    int decnInFile;       // Number of decisions in file
    int dimInFile[ndim];  // State space dimensions in file
    int status;

    // Open file for reading.

    ifstream fin( path.data() );

    // Read the policy file header.

    if ( fin.is_open() )
    {
	status = readPolicyHeader( fin, ndim, &ndimInFile, &decnInFile,
				   dimInFile );
    }
    else
    {
	status = QNET_BAD_FILE;
    }
    if ( status < 0 )
    {
	fin.close();
	return status;
    }

    // Check make sure we are expecting the correct number of dimensions
    // and decisions.

    if ( ndimInFile != ndim )
    {
	fin.close();
	return QNET_BAD_DIMENSION;
    }

    if ( decnInFile != decn )
    {
	fin.close();
	return QNET_BAD_DECISION;
    }

    // If there is more than one decision data element for each state
    // space location then we'll need to increase the dimension of the array
    // we read from the file by one.

    int arrayNDim = ndim + ( decn > 1 ? 1 : 0 );
    int arrayDim[arrayNDim];
    for ( int i = 0; i < ndim; i++ )
    {
	if ( dim[i] != dimInFile[i] )
	{
	    fin.close();
	    return QNET_BAD_TRUNCATION;
	}
	arrayDim[i] = dim[i] + 1;
    }
    if ( decn > 1 ) arrayDim[ndim] = decn;

    // Read in the policy data.

    readPolicyData1D( fin, arrayNDim, arrayDim, (int*) p );

    // All done

    fin.close();

    return QNET_NORMAL;
}

//----------------------------------------------------------------------------

int writePolicyFile1D( string path, int ndim, int decn, int dim[], void* p, 
		       char* comment )
//
// ---------------------------------------------------------------------------
// Writes a policy data file from policy data stored in the array pointed
// to by p that was created with the dynArray routines.
//
// Parameters:
//   string path         - name of file to write
//   int ndim            - dimension of the data set being saved
//   int decn            - number of decisions for each state
//   int dim[ndim+1]     - array of largest indices in state space
//                          (one less than state space dimensions)
//   int* p              - pointer portion of array being written
//   char* comment       - string to append to end of file
//
// Return Value:
//   int status          - zero if no error, negative if error
//
// Example Usage:
//
//   Assume that a policy array has been created with
//
//     int ndim = 3;
//     int decn = 2;
//     int N[] = { 0, 25, 25, 25 }; // N[0] is not used
//     int**** p = (int****) makePolicyArray( ndim, decn, &N[1] );
//
//   where the state space is 3-dimensional and each state will have 2
//   decision values.  Note that the created array is 3+1 = 4 dimensional.
//
//   After the policy values have been generated the policy data can be
//   written to the file with
//
//     writePolicyFile( string("policy.u"), ndim, decn, &N[1], p,
//                      "Policy data" );
//
//   If no comment is desired, then NULL can be passed as the last parameter.
// ---------------------------------------------------------------------------
//
{
    ofstream fout( path.data() );

    // Make sure we hopend the file...

    if ( ! fout.is_open() ) return QNET_BAD_FILE;

    // Write header to file

    writePolicyHeader( fout, ndim, decn, dim );

    // Compute array dimensions from state space dimensions; each array
    // dimension is one more than the largest index used to access the
    // corresponding dimension in the state space.  We'll go ahead and store
    // the decision dimension as well (this does not need to be adjusted)
    // but it will be ignored unless decn > 1.
  
    int arrayNDim = ndim + ( decn > 1 ? 1 : 0 );
    int arrayDim[arrayNDim];
    for ( int i = 0; i < ndim; i++ )
    {
	arrayDim[i] = dim[i] + 1;
    }
    if ( decn > 1 ) arrayDim[ndim] = decn;

    // If the number of decisions possible in a state is greater than 1 then
    // the data we should save is stored in an ndim+1 dimensional array.

    writePolicyData1D( fout, arrayNDim, arrayDim, (int*) p );

    // Write out comment

    writeComment( fout, comment );

    // Done writing, close the file.

    fout.close();

    return QNET_NORMAL;
}

//----------------------------------------------------------------------------

static void readPolicyData( ifstream& fin, int ndim, int dim[], int* p )
//
// ---------------------------------------------------------------------------
// This is a RECURSIVE function.
//
// Reads the data portion of a policy file.  The array to contain the
// policy data is assumed to have been created using the dynArray routines.
//
// Parameters:
//   ifstream& fin       - file stream to read data from
//   int ndim            - dimension of the array being read
//   int dim[ndim]       - array of largest indices in state space
//                          (one less than state space dimensions)
//   int* p              - pointer portion of array being read
//
// Return Value:
//   none
// ---------------------------------------------------------------------------
//
{
    if ( ndim == 1 )
    {
	for ( int i = 0; i < dim[0]; i++ )
	{
	    fin >> p[i];
	}
    }
    else if ( ndim > 1 )
    {
	ndim--;
	for ( int i = 0; i < dim[0]; i++ )
	{
	    readPolicyData( fin, ndim, &dim[1], ((int**)p)[i] );
	}
    }
}

//----------------------------------------------------------------------------

static void writePolicyData( ofstream& fout, int ndim, int dim[], int* p )
//
// ---------------------------------------------------------------------------
// This is a RECURSIVE function.
//
// Writes the data portion of a policy file.  The array containing the
// policy data is assumed to have been created using the dynArray routines.
//
// Parameters:
//   ofstream& fout      - file stream to write data to
//   int ndim            - dimension of the array being written
//   int dim[ndim]       - array of largest indices in state space
//                          (one less than state space dimensions)
//   int* p              - pointer portion of array being written
//
// Return Value:
//   none
// ---------------------------------------------------------------------------
//
{
    if ( ndim == 1 )
    {
	for ( int i = 0; i < dim[0]; i++ )
	{
	    fout << ' ' << p[i];
	}
    }
    else if ( ndim > 1 )
    {
	ndim--;
	for ( int i = 0; i < dim[0]; i++ )
	{
	    writePolicyData( fout, ndim, &dim[1], ((int**)p)[i] );
	}
    }
}

//----------------------------------------------------------------------------

int readPolicyFile( string path, int ndim, int decn, int dim[], void* p )
//
// ---------------------------------------------------------------------------
// Reads a policy data file into a policy array created with the dynArray
// routines.
//
// Parameters:
//   string path         - name of file to read from
//   int ndim            - dimension of the array being read
//   int decn            - number of decision for each state
//   int dim[ndim+1]     - array of largest indices in state space
//                          (one less than state space dimensions)
//   void* p             - pointer to policy data array
//
// Return Value:
//   int status          - zero if no error, negative if error
//
// Example Usage:
//
//   To read a policy file, the dimension and decision information must
//   already be known and the array to hold the policy data must already be
//   declared.  The information necessary to do this can be obtained with
//   the getPolicyFileInfo() function.
//
//     int maxDim = 3; // largest possible dimension of data set state space
//     int ndim;
//     int decn;
//     int N[maxDim + 1];
//     getPolicyFileInfo( string("policy.u"), maxDim, &ndim, &decn, &N[1] );
//
//   Convention used in most of the QNET DP programs says that the truncation
//   values are indexed starting with 1.  We can follow this convention by
//   declaring N[] to have one extra element and passing getPolicyFileInfo()
//   the address of the N[1] element.
//
//   Let's assume that ndim = 3 and decn = 2.  Since decn > 1 we will need
//   to allocate an ndim+1 = 4 dimensional array to hold the policy data.
//   This is done with
//
//      int**** p = (int****) makePolicyArray( ndim, decn, &N[1] );
//
//   Note that the only change necessary to the above line when ndim and decn
//   have different values is the number of "*" used (i.e. the dimension of
//   the array).
//
//   Finally, we're ready to read the data:
//
//      readPolicyFile( string("policy.u"), ndim, decn, &N[1], p );
// ---------------------------------------------------------------------------
//
{
    int ndimInFile;       // Number of state space dimensions in file
    int decnInFile;       // Number of decisions in file
    int dimInFile[ndim];  // State space dimensions in file
    int status;

    // Open file for reading.

    ifstream fin( path.data() );

    // Read the policy file header.

    if ( fin.is_open() )
    {
	status = readPolicyHeader( fin, ndim, &ndimInFile, &decnInFile,
				   dimInFile );
    }
    else
    {
	status = QNET_BAD_FILE;
    }
    if ( status < 0 )
    {
	fin.close();
	return status;
    }

    // Check make sure we are expecting the correct number of dimensions
    // and decisions.

    if ( ndimInFile != ndim )
    {
	fin.close();
	return QNET_BAD_DIMENSION;
    }

    if ( decnInFile != decn )
    {
	fin.close();
	return QNET_BAD_DECISION;
    }

    // If there is more than one decision data element for each state
    // space location then we'll need to increase the dimension of the array
    // we read from the file by one.

    int arrayNDim = ndim + ( decn > 1 ? 1 : 0 );
    int arrayDim[arrayNDim];
    for ( int i = 0; i < ndim; i++ )
    {
	if ( dim[i] != dimInFile[i] )
	{
	    fin.close();
	    return QNET_BAD_TRUNCATION;
	}
	arrayDim[i] = dim[i] + 1;
    }
    if ( decn > 1 ) arrayDim[ndim] = decn;

    // Read in the policy data.

    readPolicyData( fin, arrayNDim, arrayDim, (int*) p );

    // All done

    fin.close();

    return QNET_NORMAL;
}

//----------------------------------------------------------------------------

int writePolicyFile( string path, int ndim, int decn, int dim[], void* p, 
		     char* comment )
//
// ---------------------------------------------------------------------------
// Writes a policy data file from policy data stored in the array pointed
// to by p that was created with the dynArray routines.
//
// Parameters:
//   string path         - name of file to write
//   int ndim            - dimension of the data set being saved
//   int decn            - number of decisions for each state
//   int dim[ndim+1]     - array of largest indices in state space
//                          (one less than state space dimensions)
//   int* p              - pointer portion of array being written
//   char* comment       - string to append to end of file
//
// Return Value:
//   int status          - zero if no error, negative if error
//
// Example Usage:
//
//   Assume that a policy array has been created with
//
//     int ndim = 3;
//     int decn = 2;
//     int N[] = { 0, 25, 25, 25 }; // N[0] is not used
//     int**** p = (int****) makePolicyArray( ndim, decn, &N[1] );
//
//   where the state space is 3-dimensional and each state will have 2
//   decision values.  Note that the created array is 3+1 = 4 dimensional.
//
//   After the policy values have been generated the policy data can be
//   written to the file with
//
//     writePolicyFile( string("policy.u"), ndim, decn, &N[1], p,
//                      "Policy data" );
//
//   If no comment is desired, then NULL can be passed as the last parameter.
// ---------------------------------------------------------------------------
//
{
    ofstream fout( path.data() );

    // Make sure we hopend the file...

    if ( ! fout.is_open() ) return QNET_BAD_FILE;

    // Write header to file

    writePolicyHeader( fout, ndim, decn, dim );

    // Compute array dimensions from state space dimensions; each array
    // dimension is one more than the largest index used to access the
    // corresponding dimension in the state space.  We'll go ahead and store
    // the decision dimension as well (this does not need to be adjusted)
    // but it will be ignored unless decn > 1.
  
    int arrayNDim = ndim + ( decn > 1 ? 1 : 0 );
    int arrayDim[arrayNDim];
    for ( int i = 0; i < ndim; i++ )
    {
	arrayDim[i] = dim[i] + 1;
    }
    if ( decn > 1 ) arrayDim[ndim] = decn;

    // If the number of decisions possible in a state is greater than 1 then
    // the data we should save is stored in an ndim+1 dimensional array.

    writePolicyData( fout, arrayNDim, arrayDim, (int*) p );

    // Write out comment

    writeComment( fout, comment );

    // Done writing, close the file.

    fout.close();

    return QNET_NORMAL;
}

//----------------------------------------------------------------------------

// End of file

