// $Id: qnetUtility.cc,v 1.4 2009/08/03 17:33:21 senning Exp $
//
// Copyright (c) 2009 Jonathan Senning <jonathan.senning@gordon.edu>
// Gordon College, Department of Mathematics and Computer Science
//
// Collection of utility classes and functions for use by QNET programs
//
// ---------------------------------------------------------------------------
// Change Log
//
// 2009-06-10 Jonathan Senning <jonathan.senning@gordon.edu>
// - Original Version

#include <iostream>
#include <sstream>
#include <string>
#include "qnet.h"
#include <sys/param.h>
#include <sys/times.h>

//----------------------------------------------------------------------------

#ifndef HZ // often defined by system; on SuSE Linux 11 defined to be 100
#define HZ sysconf( _SC_CLK_TCK )
#endif

double qnetTimer( void )
// Returns the number of seconds since some fixed arbitrary time in
// the past.  The resolution of this function is about 10 milliseconds.
{
    struct tms buf;
    return double( times( &buf ) ) / double( HZ );
}

//----------------------------------------------------------------------------

void qnetVersion( const char* programNname, const char* revision,
		  const char* copyright )
//
// ---------------------------------------------------------------------------
// Display program version and copyright information
// ---------------------------------------------------------------------------
//
{
    using namespace std;

    // Begin by displaying program name and version number.  The name is
    // supplied as a parameter.  If "revision" begins with '$' then we
    // assume it is an RCS that has the form "$Revision: 1.4 $" otherwise
    // we assume it is merely a version number string.

    cout << programNname << " Version ";

    char* p = (char*) revision;
    if ( *p == '$' )
    {
	// RCS tag; skip over "$Revision: "
	while ( *p != ' ' && *p != 0 ) p++;  // skip to first white space
	p++;                                 // skip over white space
    }
    while ( *p != ' ' && *p != 0 ) cout << *p++; // display version
    cout << endl;

    // Done with version number; print copyright information

    cout << copyright << endl;
}

//----------------------------------------------------------------------------

char *qnetShortOpts( const struct option* longOpts, char* shortOpts, int n )
//
// ---------------------------------------------------------------------------
// Generate a list of short options for getopt() based on the descriptions
// of options in the long option structure used for getopt_long().  Using
// this function to generate the short list allows all the information about
// available options to be specified in one place.
//
// To avoid invalid memory access, the maximum length of the null-terminated
// short option string (including the NULL character) is given as the third
// parameter.
// ---------------------------------------------------------------------------
//
{
    char* s = shortOpts;
    struct option* p = (struct option*) longOpts;
    while ( p->name && n > 0 )
    {
	if ( p->flag == (int*) 0 )
	{
	    if ( --n > 0 ) *s++ = p->val;
	    if ( --n > 0 && p->has_arg != 0 ) *s++ = ':';
	}
	p++;
    }
    *s = 0;            // null-terminate string
    return shortOpts;
}

//----------------------------------------------------------------------------

void qnetError( int errno, std::string msg  )
//
// ---------------------------------------------------------------------------
// Many QNET functions return nonzero integer values to indicate that an
// error has occurred.  The qnetlibError() function accepts one of these
// values in errno and displays a corresponding error message.  If msg
// is not empty then the string it points to is displayed first, followed by
// the error message.
// ---------------------------------------------------------------------------
//
{
    using namespace std;

    cerr << "QNET";
    if ( ! msg.empty() )
    {
	cerr << " [" << msg << "] ";
    }
    else
    {
	cerr << ": ";
    }

    switch ( errno )
    {
	case QNET_NORMAL:
	    cerr << "normal" << endl;
	    break;
	case QNET_BAD_MAGIC:
	    cerr << "bad magic number in file" << endl;
	    break;
	case QNET_BAD_DIMENSION:
	    cerr << "invalid data set dimension" << endl;
	    break;
	case QNET_BAD_DECISION:
	    cerr << "invalid number of decisions" << endl;
	    break;
	case QNET_BAD_NVALUES:
	    cerr << "invalid number of values at each state space location"
		 << endl;
	    break;
	case QNET_BAD_TRUNCATION:
	    cerr << "invalid data set truncation" << endl;
	    break;
	case QNET_BAD_FILE:
	    cerr << "could not open file" << endl;
	    break;
	default:
	    cerr << "unknown error: " << errno << endl;
	    break;
    }
}

//----------------------------------------------------------------------------

// End of file
