// $Id: readTagged.cc,v 1.6 2008/06/03 15:31:37 senning Exp $
//
// Copyright (c) 2006 Nathan Walker <nathan.walker@gordon.edu>
// Gordon College, Department of Mathematics and Computer Science
// Extended and maintained by Jonathan Senning <jonathan.senning@gordon.edu>
//
// Date started: July 24, 2006
//
// Attempt at a generic .ini file reader.  Reads ints, floats, and doubles
// from an ASCII file.
//
// Methods input the filename and the tag name.  The supplied error value
// will be returned if the file or tag can't be found.
//
// 2007-09-19 Modified by Jonathan Senning <jonathan.senning@gordon.edu>
// - remove whitespace and comments before processing
// - introduced errval
//
// 2007-09-27 Jonathan Senning <jonathan.senning@gordon.edu>
// - Changed string parameters to char* parameters
// - Changed names to include "Tagged"
// - Added routine to read tagged floats
// - Added issep() function
// - Changed name of file from "iniread.cc" to "readtagged.cc"
//
// 2007-10-02 Jonathan Senning <jonathan.senning@gordon.edu>
// - Changed "param" to "tag"
// - Changed "errval" to "deflt"
// - Wrote man page
//
// 2008-06-03 Jonathan Senning <jonathan.senning@gordon.edu>
// - Changed "deflt" to "defaultValue"
// - introduced template function to handle all cases
// - used function overload so single function name works for multiple types

#include <fstream>
#include <string>
#include <sstream>
using namespace std;
#include "readTagged.h"

//----------------------------------------------------------------------------

static string collapse( string s )
//
// ---------------------------------------------------------------------------
// Remove whitespace (spaces and tabs) and comments from a string.
//
// Parameters:
//   string s       - string to process
//
// Return value:
//   string t       - input string with whitespace removed.
// ---------------------------------------------------------------------------
//
{
    string t = "";

    // remove whitespace

    string::iterator my_iter;
    for( my_iter = s.begin(); my_iter != s.end(); my_iter++)
    {
	if ( *my_iter != ' ' && *my_iter != '\t' )
	{
	    t = t + *my_iter;
	}
    }

    // strip out comments

    unsigned long int commentPosition = t.find( '#', 0 );
    if ( commentPosition != string::npos )
    {
	t = t.substr( 0, commentPosition );
    }

    return t;
}

//----------------------------------------------------------------------------

static bool isSeparator( char c )
//
// ---------------------------------------------------------------------------
// Returns true if argument is in the list of recognized separators.
//
// Parameter:
//   char c         - character to test
//
// Return value:
//   true           - if c is in the list of separators, otherwise false.
// ---------------------------------------------------------------------------
//
{
    const char* separators = ":=-";

    char* p = (char*) separators;
    while ( *p )
    {
	if ( c == *p++ ) return true;
    }
    return false;
}

//----------------------------------------------------------------------------

template <class T>
static T readTaggedValue( string fname, string tag, const T& defaultValue )
//
// ---------------------------------------------------------------------------
// Read a tagged value from a file.
//
// Parameters
//   string fname          - name of file to read from
//   string tag            - name of parameter to find value of
//   const T& defaultValue - value returned if requested tag not found
//
// Return value:
//   double parameter      - Either the value that was read or defaultValue
//
// ---------------------------------------------------------------------------
//
{
    T parameter = defaultValue;

    string line;
    char c;

    ifstream inifile( fname.data() );
    if ( inifile.is_open() )
    {
	getline( inifile, line );
	while ( ! inifile.eof() )
	{
	    line = collapse( line );

	    // Search for parameter name,value pair

	    if ( line.find( tag, 0 ) != string::npos )
	    {
		istringstream ins;
		ins.str( line.substr( tag.length() ) );
		ins >> c;
		if ( isSeparator( c ) )
		{
		    ins >> parameter;
		}
	    }
	    getline( inifile, line );
	}
	inifile.close();
    }

    return parameter;
}

//----------------------------------------------------------------------------

int readTagged( string fname, string tag, int defaultValue )
//
// ---------------------------------------------------------------------------
// Read a tagged integer from a file.
//
// Parameters
//   string fname        - name of file to read from
//   string tag          - name of parameter to find value of
//   int    defaultValue - value returned if requested tag not found
//
// Return value:
//   int parameter       - Either the integer that was read or defaultValue
// ---------------------------------------------------------------------------
//
{
    return readTaggedValue( fname, tag, defaultValue );
}

//----------------------------------------------------------------------------

float readTagged( string fname, string tag, float defaultValue )
//
// ---------------------------------------------------------------------------
// Read a tagged float from a file.
//
// Parameters
//   string fname        - name of file to read from
//   string tag          - name of parameter to find value of
//   float  defaultValue - value returned if requested tag not found
//
// Return value:
//   float parameter     - Either the float that was read or defaultValue
// ---------------------------------------------------------------------------
//
{
    return readTaggedValue( fname, tag, defaultValue );
}

//----------------------------------------------------------------------------

double readTagged( string fname, string tag, double defaultValue )
//
// ---------------------------------------------------------------------------
// Read a tagged double from a file.
//
// Parameters
//   string fname        - name of file to read from
//   string tag          - name of parameter to find value of
//   double defaultValue - value returned if requested tag not found
//
// Return value:
//   double parameter    - Either the double that was read or defaultValue
// ---------------------------------------------------------------------------
//
{
    return readTaggedValue( fname, tag, defaultValue );
}

//----------------------------------------------------------------------------

// End of file
