// $Id: testPolicyIO.cc,v 1.14 2009/07/02 01:51:16 qnet Exp $
//
// Copyright (c) 2007 Jonathan Senning <jonathan.senning@gordon.edu>
// Gordon College, Department of Mathematics and Computer Science
//
// 2007-09-25
//
// ---------------------------------------------------------------------------
// Change Log
//
// 2008-06-02 Jonathan Senning <jonathan.senning@gordon.edu>
// - changed file names from char* to string

#include <iostream>
#include <fstream>
#include <string>
#include <stdlib.h>
#include <time.h>
#include <qnet.h>

using namespace std;

//----------------------------------------------------------------------------

void testPolicyWrite( string filename, int ndim, int decn )
{
    cout << "Writing policy file " << filename << endl;

    int N[] = { 0, 2, 4, 3 };        // N[0] is unused

    // As this is a test procedure we want to allow for some different
    // configurations.  To do this, we'll declare policy arrays that are
    // 2, 3, and 4 dimensional.  This will allow us to handle the cases
    // of ndim = 2 and ndim = 3 along with decn = 1 or decn > 1.

    int**   p2 = NULL;
    int***  p3 = NULL;
    int**** p4 = NULL;

    // Determine the physical dimension of the array we want to allocate

    int arrayNDim = ndim + ( decn > 1 ? 1 : 0 );

    switch ( arrayNDim )
    {
	case 2:
	    p2 = (int**) makePolicyArray( ndim, decn, &N[1] );
	    break;
	case 3:
	    p3 = (int***) makePolicyArray( ndim, decn, &N[1] );
	    break;
	case 4:
	    p4 = (int****) makePolicyArray( ndim, decn, &N[1] );
	    break;
	default:
	    cerr << "Invalid dimension/decision pair" << endl;
	    return;
    }

    // We're not really going to store valid policy data, so we're going to
    // generate some random values...

    srandom( (int) ( time( (time_t*) 0 ) ) );

    // Fill up the policy data structure.  This is more complicated than it
    // would be in actual DP code because the loop below handles all valid
    // (ndim,decn) pairs (ndim = 2 or 3, decn = 1 or decn > 1).

    for ( int x1 = 0; x1 <= N[1]; x1++ )
    {
	for ( int x2 = 0; x2 <= N[2]; x2++)
	{
	    if ( ndim == 2 )
	    {
		if ( decn == 1 )
		{
		    p2[x1][x2] = random() % ( ndim + 1 );
		}
		else
		{
		    for ( int i = 0; i < decn; i++ )
		    {
			p3[x1][x2][i] = random() % ( ndim + 1 );
		    }
		}
	    }
	    else if ( ndim == 3 )
	    {
		for ( int x3 = 0; x3 <= N[3]; x3++ )
		{
		    if ( decn == 1 )
		    {
			p3[x1][x2][x3] = random() % ( ndim + 1 );
		    }
		    else
		    {
			for ( int i = 0; i < decn; i++ )
			{
			    p4[x1][x2][x3][i] = random() % ( ndim + 1 );
			}
		    }
		}
	    }
	}
    }

    // Write the policy data to a file and release the arrays

    int status = 0;
    switch ( arrayNDim )
    {
	case 2:
	    status = writePolicyFile( filename, ndim, decn, &N[1], p2, NULL );
	    destroyPolicyArray( p2, ndim, decn, &N[1] );
	    break;
	case 3:
	    status = writePolicyFile( filename, ndim, decn, &N[1], p3, NULL );
	    destroyPolicyArray( p3, ndim, decn, &N[1] );
	    break;
	case 4:
	    status = writePolicyFile( filename, ndim, decn, &N[1], p4, NULL );
	    destroyPolicyArray( p4, ndim, decn, &N[1] );
	    break;
    }
    if ( status < 0 )
    {
	qnetError( status, filename );
	return;
    }
}

//----------------------------------------------------------------------------

void testPolicyRead( string filename )
{
    int status;

    cout << "Reading policy file " << filename << endl;

    int maxdim = 3;     // largest number of dimensions expected in data set
    int ndim;           // dimension of state space
    int decn;           // number of decisions
    int N[maxdim + 1];  // array of state space truncations

    // Allow for 2, 3, or 4 dimensional policy data.  This allows for
    // ndim = 2 or 3 and decn = 1 or decn > 1.

    int**   p2 = NULL;
    int***  p3 = NULL;
    int**** p4 = NULL;

    // Read the header data from the policy file to determine the state
    // space dimension (ndim), the number of decisions for each state (decn),
    // and the truncations for each dimension (N[]).  N[0] is not used;
    // this follows the convention used in most of the QNET DP code.  

    status = getPolicyFileInfo( filename, maxdim, &ndim, &decn, &N[1] );
    if ( status  < 0 )
    {
	qnetError( status, filename );
	return;
    }

    // Based on the information read from the policy file header, we can now
    // allocate memory for the policy data and read it in.

    int arrayNDim = ndim + ( decn > 1 ? 1 : 0 );

    switch ( arrayNDim )
    {
	case 2:
	    p2 = (int**) makePolicyArray( ndim, decn, &N[1] );
	    status = readPolicyFile( filename, ndim, decn, &N[1], p2 );
	    break;
	case 3:
	    p3 = (int***) makePolicyArray( ndim, decn, &N[1] );
	    status = readPolicyFile( filename, ndim, decn, &N[1], p3 );
	    break;
	case 4:
	    p4 = (int****) makePolicyArray( ndim, decn, &N[1] );
	    status = readPolicyFile( filename, ndim, decn, &N[1], p4 );
	    break;
	default:
	    cerr << "Invalid dimension/decision pair" << endl;
	    return;
    }

    if ( status < 0 )
    {
	qnetError( status, filename );
	return;
    }

    // Report what we found...

    cout << "Dimension = " << ndim << endl;
    cout << "Decisions = " << decn << endl;
    cout << "State space truncations: ";
    for ( int i = 1; i <= ndim; i++ )
    {
	cout << ' ' << N[i];
    }
    cout << endl;

    for ( int x1 = 0; x1 <= N[1]; x1++ )
    {
	for ( int x2 = 0; x2 <= N[2]; x2++)
	{
	    if ( ndim == 2 )
	    {
		if ( decn == 1 )
		{
		    cout << ' ' << p2[x1][x2];
		}
		else
		{
		    cout << '(';
		    for ( int i = 0; i < decn; i++ )
		    {
			cout << ' ' << p3[x1][x2][i];
		    }
		    cout << " )";
		}
	    }
	    else if ( ndim == 3 )
	    {
		for ( int x3 = 0; x3 <= N[3]; x3++ )
		{
		    if ( decn == 1 )
		    {
			cout << ' ' << p3[x1][x2][x3];
		    }
		    else
		    {
			cout << '(';
			for ( int i = 0; i < decn; i++ )
			{
			    cout << ' ' << p4[x1][x2][x3][i];
			}
			cout << " )";
		    }
		}
		cout << endl;
	    }
	}
	cout << endl;
    }

    // All done, so we can release the array holding the policy data

    switch ( arrayNDim )
    {
	case 2:
	    destroyPolicyArray( p2, ndim, decn, &N[1] );
	    break;
	case 3:
	    destroyPolicyArray( p3, ndim, decn, &N[1] );
	    break;
	case 4:
	    destroyPolicyArray( p4, ndim, decn, &N[1] );
	    break;
    }
}

//----------------------------------------------------------------------------

int main( int argc, char* argv[] )
{
    cout << endl;
    cout << "==== Testing Policy IO Routines ====";
    cout << "==========================================" << endl;
    cout << endl;

    testPolicyWrite( string( "dim2decn1.policy" ), 2, 1 );
    testPolicyWrite( string( "dim2decn2.policy" ), 2, 2 );
    testPolicyWrite( string( "dim3decn1.policy" ), 3, 1 );
    testPolicyWrite( string( "dim3decn2.policy" ), 3, 2 );
    testPolicyWrite( string( "dim3decn3.policy" ), 3, 3 );

    cout << "----------------------------------------------------" << endl;
    cout << "This test should fail as we will try and write to a"
         << " nonexistent directory..." << endl;
    testPolicyWrite( string( "/nonexistent/directory/junk.policy" ), 2, 1 );
    cout << "----------------------------------------------------" << endl;
    cout << "This test should fail as we will try and read from a"
         << " nonexistent file..." << endl;
    testPolicyRead( string( "/nonexistent/directory/junk.policy" ) );
    cout << "----------------------------------------------------" << endl;

    cout << "----------------------------------------------------" << endl;
    testPolicyRead( string( "dim2decn1.policy" ) );
    cout << "----------------------------------------------------" << endl;
    testPolicyRead( string( "dim2decn2.policy" ) );
    cout << "----------------------------------------------------" << endl;
    testPolicyRead( string( "dim3decn1.policy" ) );
    cout << "----------------------------------------------------" << endl;
    testPolicyRead( string( "dim3decn2.policy" ) );
    cout << "----------------------------------------------------" << endl;
    testPolicyRead( string( "dim3decn3.policy" ) );
    cout << "----------------------------------------------------" << endl;

    return 0;
}
