// $Id: testPolicyIO_1D.cc,v 1.2 2009/07/02 01:50:21 qnet Exp $
//
// Copyright (c) 2007-2009 Department of Mathematics and Computer Science
// Gordon College, 255 Grapevine Road, Wenham, MA 01984
//
// Author:  Jonathan Senning <jonathan.senning@gordon.edu>
// Written: 2007-09-25
//
// ---------------------------------------------------------------------------
// Change Log
//
// 2008-06-02 Jonathan Senning <jonathan.senning@gordon.edu>
// - changed file names from char* to string
//
// 2009-06-23 Jonathan Senning <jonathan.senning@gordon.edu>
// - Use MakeIndex and work with single dimensional policy array

#include <iostream>
#include <fstream>
#include <string>
#include <stdlib.h>
#include <time.h>
#include <qnet.h>
#include <MultiIndex.h>

using namespace std;

//----------------------------------------------------------------------------

void testPolicyWrite( string filename, int ndim, int decn )
{
    cout << "Writing policy file " << filename << endl;

    int N[] = { 0, 2, 4, 3 };        // N[0] is unused

    // As this is a test procedure we want to allow for some different
    // configurations.  To do this, we'll declare policy arrays that are
    // 2, 3, and 4 dimensional.  This will allow us to handle the cases
    // of ndim = 2 and ndim = 3 along with decn = 1 or decn > 1.

    int* p = NULL;

    // Determine the physical dimension of the array we want to allocate

//    int arrayNDim = ndim + ( decn > 1 ? 1 : 0 );

    p = makePolicyArray1D( ndim, decn, &N[1] );

    // We're not really going to store valid policy data, so we're going to
    // generate some random values...

    srandom( (int) ( time( (time_t*) 0 ) ) );

    // Fill up the policy data structure.  This is more complicated than it
    // would be in actual DP code because the loop below handles all valid
    // (ndim,decn) pairs (ndim = 2 or 3, decn = 1 or decn > 1).

    if ( decn > 1 )
    {
	int arrayDim[ndim + 1];
	for ( int i = 0; i < ndim; i++ )
	{
	    arrayDim[i] = N[i+1];
	}
	arrayDim[ndim] = decn - 1;
	MultiIndex state( ndim + 1, arrayDim );
	for ( int X = state.start( 1 ); state.inRange(); X = state.next( 1 ) )
	{
	    cout << "State: " << state.toString() << ':';
	    for ( int i = 0; i < decn; i++ )
	    {
		cout << ' ' << X + i;
		p[X + i] = random() % ( ndim + 1 );
	    }
	    cout << endl;
	}
    }
    else if ( decn == 1 )
    {
	MultiIndex state( ndim, &N[1] );
	for ( int X = state.start(); state.inRange(); X = state.next() )
	{
	    p[X] = random() % ( ndim + 1 );
	}
    }

    // Write the policy data to a file and release the arrays

    int status = 0;
    status = writePolicyFile1D( filename, ndim, decn, &N[1], p, NULL );
    destroyPolicyArray1D( p, ndim, decn, &N[1] );
    if ( status < 0 )
    {
	qnetError( status, filename );
	return;
    }
}

//----------------------------------------------------------------------------

void testPolicyRead( string filename )
{
    int status;

    cout << "Reading policy file " << filename << endl;

    int maxdim = 3;     // largest number of dimensions expected in data set
    int ndim;           // dimension of state space
    int decn;           // number of decisions
    int N[maxdim + 1];  // array of state space truncations

    // Allow for 2, 3, or 4 dimensional policy data.  This allows for
    // ndim = 2 or 3 and decn = 1 or decn > 1.

    int* p = NULL;

    // Read the header data from the policy file to determine the state
    // space dimension (ndim), the number of decisions for each state (decn),
    // and the truncations for each dimension (N[]).  N[0] is not used;
    // this follows the convention used in most of the QNET DP code.  

    status = getPolicyFileInfo( filename, maxdim, &ndim, &decn, &N[1] );
    if ( status  < 0 )
    {
	qnetError( status, filename );
	return;
    }

    // Report what we found...

    cout << "Dimension = " << ndim << endl;
    cout << "Decisions = " << decn << endl;
    cout << "State space truncations: ";
    for ( int i = 1; i <= ndim; i++ )
    {
	cout << ' ' << N[i];
    }
    cout << endl;

    // Based on the information read from the policy file header, we can now
    // allocate memory for the policy data and read it in.

//    int arrayNDim = ndim + ( decn > 1 ? 1 : 0 );
    p = makePolicyArray1D( ndim, decn, &N[1] );

    status = readPolicyFile1D( filename, ndim, decn, &N[1], p );
    if ( status < 0 )
    {
	qnetError( status, filename );
	return;
    }

    if ( decn > 1 )
    {
	int arrayDim[ndim + 1];
	for ( int i = 0; i < ndim; i++ )
	{
	    arrayDim[i] = N[i+1];
	}
	arrayDim[ndim] = decn - 1;
	MultiIndex state( ndim + 1, arrayDim );
	int* x = state.getCounterArray( 1 );
	for ( int X = state.start( 1 ); state.inRange(); X = state.next( 1 ) )
	{
	    cout << '(';
	    for ( int i = 0; i < decn; i++ )
	    {
		cout << ' ' << p[X + i];
	    }
	    cout << ')';
	    if ( x[ndim] == N[ndim] ) cout << endl;
	}
    }
    else if ( decn == 1 )
    {
	MultiIndex state( ndim, &N[1] );
	int* x = state.getCounterArray( 1 );
	for ( int X = state.start(); state.inRange(); X = state.next() )
	{
	    cout << ' ' << p[X];
	    if ( x[ndim] == N[ndim] ) cout << endl;
	}
    }

    // All done, so we can release the array holding the policy data

    destroyPolicyArray1D( p, ndim, decn, &N[1] );
}

//----------------------------------------------------------------------------

int main( int argc, char* argv[] )
{
    cout << endl;
    cout << "==== Testing Policy IO Routines ====";
    cout << "==========================================" << endl;
    cout << endl;

    testPolicyWrite( string( "dim2decn1.policy" ), 2, 1 );
    testPolicyWrite( string( "dim2decn2.policy" ), 2, 2 );
    testPolicyWrite( string( "dim3decn1.policy" ), 3, 1 );
    testPolicyWrite( string( "dim3decn2.policy" ), 3, 2 );
    testPolicyWrite( string( "dim3decn3.policy" ), 3, 3 );

    cout << "----------------------------------------------------" << endl;
    cout << "This test should fail as we will try and write to a"
         << " nonexistent directory..." << endl;
    testPolicyWrite( string( "/nonexistent/directory/junk.policy" ), 2, 1 );
    cout << "----------------------------------------------------" << endl;
    cout << "This test should fail as we will try and read from a"
         << " nonexistent file..." << endl;
    testPolicyRead( string( "/nonexistent/directory/junk.policy" ) );
    cout << "----------------------------------------------------" << endl;

    cout << "----------------------------------------------------" << endl;
    testPolicyRead( string( "dim2decn1.policy" ) );
    cout << "----------------------------------------------------" << endl;
    testPolicyRead( string( "dim2decn2.policy" ) );
    cout << "----------------------------------------------------" << endl;
    testPolicyRead( string( "dim3decn1.policy" ) );
    cout << "----------------------------------------------------" << endl;
    testPolicyRead( string( "dim3decn2.policy" ) );
    cout << "----------------------------------------------------" << endl;
    testPolicyRead( string( "dim3decn3.policy" ) );
    cout << "----------------------------------------------------" << endl;

    return 0;
}
