// $Id: testValueIO.cc,v 1.12 2009/07/02 01:51:32 qnet Exp $
//
// Copyright (c) 2007 Jonathan Senning <jonathan.senning@gordon.edu>
// Gordon College, Department of Mathematics and Computer Science
//
// 2007-09-25
//
// ---------------------------------------------------------------------------
// Change Log
//
// 2008-06-02 Jonathan Senning <jonathan.senning@gordon.edu>
// - changed file names from char* to string

#include <iostream>
#include <fstream>
#include <string>
#include <qnet.h>

using namespace std;

//----------------------------------------------------------------------------

void testValueWrite( string filename, int ndim, int nval )
{
    cout << "Writing value file " << filename << endl;

    int N[] = { 0, 2, 4, 3 };        // N[0] is unused

    // As this is a test procedure we want to allow for some different
    // configurations.  To do this, we'll declare value arrays that are
    // 2 and 3 dimensional.

    double**   h2 = NULL;
    double***  h3 = NULL;
    double**** h4 = NULL;

    // Determine the physical dimension of the array we want to allocate

    int arrayNDim = ndim + ( nval > 1 ? 1 : 0 );

    switch ( arrayNDim )
    {
	case 2:
	    h2 = (double**) makeValueArray( ndim, nval, &N[1] );
	    break;
	case 3:
	    h3 = (double***) makeValueArray( ndim, nval, &N[1] );
	    break;
	case 4:
	    h4 = (double****) makeValueArray( ndim, nval, &N[1] );
	    break;
	default:
	    cerr << "Invalid dimension/nvalue pair" << endl;
	    return;
    }

    // We're not really going to store real value data, so we're going to
    // generate some values...

    // Fill up the value data structure.  This is more complicated than it
    // would be in actual DP code because the loop below handles all valid
    // (ndim,nval) pairs (ndim = 2 or 3, nval = 1 or nval > 1).

    for ( int x1 = 0; x1 <= N[1]; x1++ )
    {
	for ( int x2 = 0; x2 <= N[2]; x2++)
	{
	    if ( ndim == 2 )
	    {
		if ( nval == 1 )
		{
		    h2[x1][x2] = x1 + 10.0 * x2;
		}
		else
		{
		    for ( int i = 0; i < nval; i++ )
		    {
			h3[x1][x2][i] = 0.1 * ( i + 1 ) + x1 + 10.0 * x2;
		    }
		}
	    }
	    else if ( ndim == 3 )
	    {
		for ( int x3 = 0; x3 <= N[3]; x3++ )
		{
		    if ( nval == 1 )
		    {
			h3[x1][x2][x3] = x1 + 10.0 * x2 + 100.0 * x3;
		    }
		    else
		    {
			for ( int i = 0; i < nval; i++ )
			{
			    h4[x1][x2][x3][i] = 0.1 * ( i + 1 ) + x1 
				+ 10.0 * x2 + 100.0 * x3;
			}
		    }
		}
	    }
	}
    }

    // Write the value data to a file and release the arrays

    int status = 0;
    switch ( arrayNDim )
    {
	case 2:
	    status = writeValueFile( filename, ndim, nval, &N[1], h2, NULL );
	    destroyValueArray( h2, ndim, nval, &N[1] );
	    break;
	case 3:
	    status = writeValueFile( filename, ndim, nval, &N[1], h3,
				     (char*) "3D Data" );
	    destroyValueArray( h3, ndim, nval, &N[1] );
	    break;
	case 4:
	    status = writeValueFile( filename, ndim, nval, &N[1], h4,
				     (char*) "More 3D Data" );
	    destroyValueArray( h4, ndim, nval, &N[1] );
	    break;
    }

    if ( status < 0 )
    {
	qnetError( status, filename );
    }
}

//----------------------------------------------------------------------------

void testValueRead( string filename )
{
    int status;

    cout << "Reading value file " << filename << endl;

    int maxdim = 3;     // largest number of dimensions expected in data set
    int ndim;           // dimension of state space
    int nval;           // number of values at each state space location
    int N[maxdim + 1];  // array of state space truncations

    // Allow for 2 or 3 dimensional value data.

    double**   h2 = NULL;
    double***  h3 = NULL;
    double**** h4 = NULL;

    // Read the header data from the value file to determine the state
    // space dimension (ndim) and the truncations for each dimension (N[]).
    // N[0] is not used; this follows the convention used in most of the
    // QNET DP code. 

    status = getValueFileInfo( filename, maxdim, &ndim, &nval, &N[1] );
    if ( status < 0 )
    {
	qnetError( status, filename );
	return;
    }

    // Based on the information read from the policy file header, we can now
    // allocate memory for the policy data and read it in.

    int arrayNDim = ndim + ( nval > 1 ? 1 : 0 );

    switch ( arrayNDim )
    {
	case 2:
	    h2 = (double**) makeValueArray( ndim, nval, &N[1] );
	    status = readValueFile( filename, ndim, nval, &N[1], h2 );
	    break;
	case 3:
	    h3 = (double***) makeValueArray( ndim, nval, &N[1] );
	    status = readValueFile( filename, ndim, nval, &N[1], h3 );
	    break;
	case 4:
	    h4 = (double****) makeValueArray( ndim, nval, &N[1] );
	    status = readValueFile( filename, ndim, nval, &N[1], h4 );
	    break;
	default:
	    cerr << "Invalid dimension" << endl;
	    return;
    }

    if ( status < 0 )
    {
	qnetError( status, filename );
	return;
    }

    // Report what we found...

    cout << "Dimension  = " << ndim << endl;
    cout << "Num Values = " << nval << endl;
    cout << "State space truncations: ";
    for ( int i = 1; i <= ndim; i++ )
    {
	cout << ' ' << N[i];
    }
    cout << endl;

    for ( int x1 = 0; x1 <= N[1]; x1++ )
    {
	for ( int x2 = 0; x2 <= N[2]; x2++)
	{
	    if ( ndim == 2 )
	    {
		if ( nval == 1 )
		{
		    cout << ' ' << h2[x1][x2];
		}
		else
		{
		    cout << '(';
		    for ( int i = 0; i < nval; i++ )
		    {
			cout << ' ' << h3[x1][x2][i];
		    }
		    cout << " )";
		}
	    }
	    else if ( ndim == 3 )
	    {
		for ( int x3 = 0; x3 <= N[3]; x3++ )
		{
		    if ( nval == 1 )
		    {
			cout << ' ' << h3[x1][x2][x3];
		    }
		    else
		    {
			cout << '(';
			for ( int i = 0; i < nval; i++ )
			{
			    cout << ' ' << h4[x1][x2][x3][i];
			}
			cout << " )";
		    }
		}
	    }
	    cout << endl;
	}
	cout << endl;
    }

    // All done, so we can release the array holding the policy data

    switch ( arrayNDim )
    {
	case 2:
	    destroyValueArray( h2, ndim, nval, &N[1] );
	    break;
	case 3:
	    destroyValueArray( h3, ndim, nval, &N[1] );
	    break;
	case 4:
	    destroyValueArray( h4, ndim, nval, &N[1] );
	    break;
    }
}

//----------------------------------------------------------------------------

int main( int argc, char* argv[] )
{
    cout << endl;
    cout << "==== Testing Value IO Routines ====";
    cout << "==========================================" << endl;
    cout << endl;

    testValueWrite( string( "dim2nval1.value" ), 2, 1 );
    testValueWrite( string( "dim2nval2.value" ), 2, 2 );
    testValueWrite( string( "dim3nval1.value" ), 3, 1 );
    testValueWrite( string( "dim3nval2.value" ), 3, 2 );
    testValueWrite( string( "dim3nval3.value" ), 3, 3 );

    cout << "----------------------------------------------------" << endl;
    cout << "This test should fail as we will try and write to a"
	 << " nonexistent directory..." << endl;
    testValueWrite( "/nonexistent/directory/junk.value", 2, 1 );
    cout << "----------------------------------------------------" << endl;
    cout << "This test should fail as we will try and read from a"
	 << " nonexistent file..." << endl;
    testValueRead( "/nonexistent/directory/junk.value" );
    cout << "----------------------------------------------------" << endl;

    cout << "----------------------------------------------------" << endl;
    testValueRead( string( "dim2nval1.value" ) );
    cout << "----------------------------------------------------" << endl;
    testValueRead( string( "dim2nval2.value" ) );
    cout << "----------------------------------------------------" << endl;
    testValueRead( string( "dim3nval1.value" ) );
    cout << "----------------------------------------------------" << endl;
    testValueRead( string( "dim3nval2.value" ) );
    cout << "----------------------------------------------------" << endl;
    testValueRead( string( "dim3nval3.value" ) );
    cout << "----------------------------------------------------" << endl;

    return 0;
}
