// $Id: testValueIO_1D.cc,v 1.2 2009/07/02 01:50:46 qnet Exp $
//
// Copyright (c) 2007-2009 Department of Mathematics and Computer Science
// Gordon College, 255 Grapevine Road, Wenham, MA 01984
//
// Author:  Jonathan Senning <jonathan.senning@gordon.edu>
// Written: 2007-09-25
//
// ---------------------------------------------------------------------------
// Change Log
//
// 2008-06-02 Jonathan Senning <jonathan.senning@gordon.edu>
// - changed file names from char* to string
//
// 2009-06-23 Jonathan Senning <jonathan.senning@gordon.edu>
// - changed to use MultiIndex and use single dimensional value array

#include <iostream>
#include <fstream>
#include <string>
#include <qnet.h>
#include <MultiIndex.h>

using namespace std;

//----------------------------------------------------------------------------

void testValueWrite( string filename, int ndim, int nval )
{
    cout << "Writing value file " << filename << endl;

    int N[] = { 0, 2, 4, 3 };        // N[0] is unused

    // Create array to hold value data

    double* h = makeValueArray1D( ndim, nval, &N[1] );

    // We're not really going to store real value data, so we're going to
    // generate some values...

    // Fill up the value data structure.  This is more complicated than it
    // would be in actual DP code because the loop below handles all valid
    // (ndim,nval) pairs (ndim = 2 or 3, nval = 1 or nval > 1).

    if ( nval == 1 )
    {
	MultiIndex Counter( ndim, &N[1] );
	int* z = Counter.getCounterArray( 1 );

	for ( int x = Counter.start(); Counter.inRange(); x = Counter.next() )
	{
	    double s = 0.0;
	    for ( int i = ndim; i >= 1; i-- )
	    {
		s = 10.0 * s + z[i];
	    }
	    h[x] = s;
	}
    }
    else
    {
	int arrayN[ndim+1];
	for ( int i = 0; i < ndim; i++ )
	{
	    arrayN[i] = N[i+1];
	}
	arrayN[ndim] = nval - 1;
	MultiIndex Counter( ndim+1, arrayN );
	int* z = Counter.getCounterArray( 1 );
	
	for ( int x = Counter.start(1); Counter.inRange(); x = Counter.next(1) )
	{
	    double s = 0.0;
	    for ( int i = ndim; i >= 1; i-- )
	    {
		s = 10.0 * s + z[i];
	    }
	    for ( int u = 0; u < nval; u++ )
	    {
		h[x + u] = s + 0.1 * ( u + 1 );
	    }
	}
    }

    // Write the value data to a file and release the arrays

    int status = 0;
    status = writeValueFile1D( filename, ndim, nval, &N[1], h, NULL );
    destroyValueArray1D( h, ndim, nval, &N[1] );
    

    if ( status < 0 )
    {
	qnetError( status, filename );
    }
}

//----------------------------------------------------------------------------

void testValueRead( string filename )
{
    int status;

    cout << "Reading value file " << filename << endl;

    int maxdim = 3;     // largest number of dimensions expected in data set
    int ndim;           // dimension of state space
    int nval;           // number of values at each state space location
    int N[maxdim + 1];  // array of state space truncations

    // Read the header data from the value file to determine the state
    // space dimension (ndim) and the truncations for each dimension (N[]).
    // N[0] is not used; this follows the convention used in most of the
    // QNET DP code. 

    status = getValueFileInfo( filename, maxdim, &ndim, &nval, &N[1] );
    if ( status < 0 )
    {
	qnetError( status, filename );
	return;
    }

    // Based on the information read from the policy file header, we can now
    // allocate memory for the policy data and read it in.

    double* h = makeValueArray1D( ndim, nval, &N[1] );
    status = readValueFile1D( filename, ndim, nval, &N[1], h );

    if ( status < 0 )
    {
	qnetError( status, filename );
	return;
    }

    // Report what we found...

    cout << "Dimension  = " << ndim << endl;
    cout << "Num Values = " << nval << endl;
    cout << "State space truncations: ";
    for ( int i = 1; i <= ndim; i++ )
    {
	cout << ' ' << N[i];
    }
    cout << endl;

    if ( nval == 1 )
    {
	MultiIndex Counter( ndim, &N[1] );
	for ( int x = Counter.start(); Counter.inRange(); x = Counter.next() )
	{
	    cout << ' ' << h[x];
	    cout << endl;
	}
    }
    else
    {
	int arrayN[ndim+1];
	for ( int i = 0; i < ndim; i++ )
	{
	    arrayN[i] = N[i+1];
	}
	arrayN[ndim] = nval - 1;
	MultiIndex Counter( ndim+1, arrayN );
	
	for ( int x = Counter.start(1); Counter.inRange(); x = Counter.next(1) )
	{
	    cout << '(';
	    for ( int u = 0; u < nval; u++ )
	    {
		cout << ' ' << h[x + u];
	    }
	    cout << " )" << endl;
	}
    }

    // All done, so we can release the array holding the policy data

    destroyValueArray1D( h, ndim, nval, &N[1] );
}

//----------------------------------------------------------------------------

int main( int argc, char* argv[] )
{
    cout << endl;
    cout << "==== Testing Value IO Routines ====";
    cout << "==========================================" << endl;
    cout << endl;

    testValueWrite( string( "dim2nval1.value" ), 2, 1 );
    testValueWrite( string( "dim2nval2.value" ), 2, 2 );
    testValueWrite( string( "dim3nval1.value" ), 3, 1 );
    testValueWrite( string( "dim3nval2.value" ), 3, 2 );
    testValueWrite( string( "dim3nval3.value" ), 3, 3 );

    cout << "----------------------------------------------------" << endl;
    cout << "This test should fail as we will try and write to a"
	 << " nonexistent directory..." << endl;
    testValueWrite( "/nonexistent/directory/junk.value", 2, 1 );
    cout << "----------------------------------------------------" << endl;
    cout << "This test should fail as we will try and read from a"
	 << " nonexistent file..." << endl;
    testValueRead( "/nonexistent/directory/junk.value" );
    cout << "----------------------------------------------------" << endl;

    cout << "----------------------------------------------------" << endl;
    testValueRead( string( "dim2nval1.value" ) );
    cout << "----------------------------------------------------" << endl;
    testValueRead( string( "dim2nval2.value" ) );
    cout << "----------------------------------------------------" << endl;
    testValueRead( string( "dim3nval1.value" ) );
    cout << "----------------------------------------------------" << endl;
    testValueRead( string( "dim3nval2.value" ) );
    cout << "----------------------------------------------------" << endl;
    testValueRead( string( "dim3nval3.value" ) );
    cout << "----------------------------------------------------" << endl;

    return 0;
}
