// $Id: valueIO.cc,v 1.12 2009/07/02 01:27:51 qnet Exp $
//
// Copyright (c) 2007-2009 Department of Mathematics and Computer Science
// Gordon College, 255 Grapevine Road, Wenham, MA 01984
//
// Author:  Jonathan Senning <jonathan.senning@gordon.edu>
// Written: 2007-09-26
//
// These functions are used to read and write value data files using the
// Nov 1, 2007 raw format:
//
//    Start         End
//   Position     Position     Type     Name     Description
//  -----------  -----------  -------  ------  -----------------------------
//      0            3        int              Magic Number; should be -7
//
//      4            7        int      ndim    Dimension of state space
//
//      8           11        int              Number of values at location;
//                                             must be 1 for now
//
//     12        12+4*ndim-1  int*     dim[]   State space truncations
//                                             (each one less than individual
//                                             state space array dimensions)
//
//  12+4*ndim       K-1       double           value/cost data in row-major
//                                             order
//
//      K        end of file  char*            comment string
//
// where
//      K = 12 + 4 * ndim + 4 * ( Prod_{i=1}^ndim (dim[i]+1) )
//
// ---------------------------------------------------------------------------
// Change Log
//
// 2008-06-02 Jonathan Senning <jonathan.senning@gordon.edu>
// - changed filename parameter from char* to string
//
// 2009-06-23 Jonathan Senning <jonathan.senning@gordon.edu>
// - value data now in single dimensional array

#include <fstream>
#include <string>
#include <time.h>
#include "qnet.h"
using namespace std;

//----------------------------------------------------------------------------

static int readValueHeader( ifstream& fin, int maxdim, int* ndim, int* nval,
			    int dim[] )
//
// ---------------------------------------------------------------------------
// Reads a value data file header.
//
// Parameters:
//   ifstream& fin       - file stream to read data from
//   int maxdim          - maximum dimension of the data set
//   int* ndim           - pointer to location to store state space dimension
//   int* nval           - pointer to location to store num values per state
//   int dim[]           - array of state space dimensions.  This array must
//                         have already been declared and contain at least
//                         maxdim elements.
//
// Return Value:
//   int status          - zero if no error, negative if error
// ---------------------------------------------------------------------------
//
{
    // Read the magic number that should be stored at the start of the file.
    // If this does not match the number used for value files then we should
    // generate a warning or an error.

    int type;
    fin.read( reinterpret_cast<char*> (&type), sizeof( int ) );
    if ( type != QNET_VALUE_MAGIC ) return QNET_BAD_MAGIC;

    // Read the dimension and the individual state space dimensions.

    fin.read( reinterpret_cast<char*> (ndim), sizeof( int ) );

    // Read the number of data values stored at each state space location.

    fin.read( reinterpret_cast<char*> (nval), sizeof( int ) );

    // Make sure that the caller has indicated that the dim[] array is large
    // enough to hold the dimension data we're going to put into it.

    if ( maxdim < 0 || *ndim > maxdim ) return QNET_BAD_DIMENSION;

    // Read in the individual dimensions.

    for ( int i = 0; i < *ndim; i++ )
    {
	fin.read( reinterpret_cast<char*> (&dim[i]), sizeof( int ) );
    }

    return QNET_NORMAL;
}

//----------------------------------------------------------------------------

static void writeValueHeader( ofstream& fout, int ndim, int nval, int dim[] )
//
// ---------------------------------------------------------------------------
// Writes a value data file header.
//
// Parameters:
//   ofstream& fout      - file stream to write data to
//   int ndim            - dimension of the array being written
//   int nval            - number of values per state space location
//   int dim[ndim]       - array of largest indices in state space
//
// Return Value:
//   none
// ---------------------------------------------------------------------------
//
{
    // Write out magic number to the file.  Only value files should have
    // this number stored in the first four bytes of the file.

    int type = QNET_VALUE_MAGIC;
    fout.write( reinterpret_cast<char*> (&type), sizeof( int ) );

    // Write dimension of state space and state space dimensions to file.

    fout.write( reinterpret_cast<char*> (&ndim), sizeof( int ) );

    // Write number of values per state space location

    fout.write( reinterpret_cast<char*> (&nval), sizeof( int ) );

    // Write individual state space dimensions to file.

    for ( int i = 0; i < ndim; i++ )
    {
	fout.write( reinterpret_cast<char*> (&dim[i]), sizeof( int ) );
    }

}

//----------------------------------------------------------------------------

static void readValueData1D( ifstream& fin, int ndim, int dim[], double* h )
//
// ---------------------------------------------------------------------------
// Reads the data portion of a value file.  The arrays to contain the
// value data is assumed to have been created using the dynArray routines.
// If g is passed as NULL pointer then we do not want its data.
//
// Parameters:
//   ifstream& fin       - file stream to read data from
//   int ndim            - dimension of the arrays being read
//   int dim[ndim]       - array of largest indices in state space
//                          (one less than state space dimensions)
//   int* h              - pointer portion of h array being read
//
// Return Value:
//   none
// ---------------------------------------------------------------------------
//
{
    int size = 1;
    for ( int i = 0; i < ndim; i++ )
    {
	size *= dim[i];
    }
    fin.read( reinterpret_cast<char*> (&h[0]), size * sizeof( double ) );
}

//----------------------------------------------------------------------------

static void writeValueData1D( ofstream& fout, int ndim, int dim[], double* h )
//
// ---------------------------------------------------------------------------
// Writes the data portion of a value file.  The array containing the
// value data is assumed to have been created using the dynArray routines.
//
// Parameters:
//   ofstream& fout      - file stream to write data to
//   int ndim            - dimension of the array being written
//   int dim[ndim]       - array of largest indices in state space
//                          (one less than state space dimensions)
//   int* h              - pointer portion of h array being written
//
// Return Value:
//   none
// ---------------------------------------------------------------------------
//
{
    int size = 1;
    for ( int i = 0; i < ndim; i++ )
    {
	size *= dim[i];
    }
    fout.write( reinterpret_cast<char*> (&h[0]), size * sizeof( double ) );
}

//----------------------------------------------------------------------------

static void writeComment( ofstream& fout, char* comment )
//
// ---------------------------------------------------------------------------
// Writes a comment string to a data file.  If comment is NULL then a
// string with a data-time stamp is written.
//
// Parameters:
//   ofstream& fout      - file stream to write data to
//   char* comment       - string to append to end of file
//
// Return Value:
//   none
// ---------------------------------------------------------------------------
//
{
    // Separate comment from data

    fout << endl;

    // Write comment or time stamp

    if ( comment != NULL )
    {
	fout << "COMMENT: " << comment << endl;
    }
    else
    {
	time_t t;
	time( &t );
	// '\n' is last character in ctime() string so endl is not needed
	fout << "COMMENT: " << "Value data created at " << ctime( &t );
    }
}

//----------------------------------------------------------------------------

int getValueFileInfo( string fname, int maxdim, int* ndim, int* nval,
		      int dim[] )
//
// ---------------------------------------------------------------------------
// Provides user access to value file header information.
//
// Parameters:
//   string fname        - name of file to read from
//   int maxdim          - maximum dimension of the data set
//   int* ndim           - pointer to location to store state space dimension
//   int* nval           - pointer to location to store num val per state
//   int dim[]           - array of state space dimensions.  This array must
//                         have already been declared and contain at least
//
// Return Value:
//   int status          - zero if no error, negative if error
// ---------------------------------------------------------------------------
//
{
    int status;
    ifstream fin( fname.data() );
    if ( fin.is_open() )
    {
	status = readValueHeader( fin, maxdim, ndim, nval, dim );
    }
    else
    {
	status = QNET_BAD_FILE;
    }
    fin.close();
    return status;
}

//----------------------------------------------------------------------------

int readValueFile1D( string fname, int ndim, int nval, int dim[], void* h )
//
// ---------------------------------------------------------------------------
// Reads a value data file into value arrays created with the dynArray
// routines.
//
// Parameters:
//   string fname        - name of file to read from
//   int ndim            - dimension of the array being read
//   int nval            - number of values per state space location
//   int dim[ndim]       - array of largest indices in state space
//                          (one less than state space dimensions)
//   void* h             - pointer to value data
//
// Return Value:
//   int status          - zero if no error, negative if error
//
// Example Usage:
//
//   To read a value file the dimension information must already be known
//   and the array or arrays to hold the value data must already be
//   declared.  The information necessary to do this can be obtained with
//   the getValueFileInfo() function.
//
//     int maxdim = 3; // largest possible dimension of data set state space
//     int ndim;
//     int nval;
//     int N[maxdim + 1];
//     getValueFileInfo( string("value.h"), maxdim, &ndim, &nval, &N[1] );
//
//   Convention used in most of the QNET DP programs says that the truncation
//   values are indexed starting with 1.  We can follow this convention by
//   declaring N[] to have one extra element and passing getValueFileInfo()
//   the address of the N[1] element.
//
//   If ndim = 3 and nval = 1 then we can allocate 3-dimensional array for h
//   with
//
//      double*** h = (double***) makeValueArray( ndim, nval, &N[1] );
//
//   Note that the only change necessary to the above line when ndim is 2
//   would be to use "**" in place of "***".  If, however, nval > 1 then
//   we also need an additional "dimension" to store the multiple values for
//   each state space location.  So, if ndim = 3 and nval = 2 the array h
//   should be created with
//
//      double**** h = (double****) makeValueArray( ndim, nval, &N[1] );
//
//   Finally, we're ready to read the data:
//
//      readValueFile( string("value.h"), ndim, nval, &N[1], h );
// ---------------------------------------------------------------------------
//
{
    int ndimInFile;        // Number of state space dimensions in file
    int nvalInFile;        // Number of values per state in file
    int dimInFile[ndim];   // State space dimensions in file
    int status;

    // Open file for reading.

    ifstream fin( fname.data() );

    // Read the value file header.

    if ( fin.is_open() )
    {
	status = readValueHeader( fin, ndim, &ndimInFile, &nvalInFile,
				  dimInFile );
    }
    else
    {
	status = QNET_BAD_FILE;
    }
    if ( status < 0 )
    {
	fin.close();
	return status;
    }

    // Check make sure we are expecting the correct number of dimensions
    // and values

    if ( ndimInFile != ndim )
    {
	fin.close();
	return QNET_BAD_DIMENSION;
    }

    if ( nvalInFile != nval )
    {
	fin.close();
	return QNET_BAD_NVALUES;
    }

    // If there is more than one value per state space location then we'll
    // need to increase the dimension of the array we read data into by one.

    int arrayNDim = ndim + ( nval > 1 ? 1 : 0 );
    int arrayDim[arrayNDim];
    for ( int i = 0; i < ndim; i++ )
    {
	if ( dim[i] != dimInFile[i] )
	{
	    fin.close();
	    return QNET_BAD_TRUNCATION;
	}
	arrayDim[i] = dim[i] + 1;
    }
    if ( nval > 1 ) arrayDim[ndim] = nval;

    // Read in the value data.

    readValueData1D( fin, arrayNDim, arrayDim, (double*) h );

    // All done

    fin.close();

    return QNET_NORMAL;
}

//----------------------------------------------------------------------------

int writeValueFile1D( string fname, int ndim, int nval, int dim[], void* h,
		      char* comment )
//
// ---------------------------------------------------------------------------
// Writes a value data file from value data stored in the array pointed
// to by p that was created with the dynArray routines.
//
// Parameters:
//   string fname        - name of file to write
//   int ndim            - dimension of the array being written
//   int nval            - number of values per state space location
//   int dim[ndim]       - array of largest indices in state space
//                          (one less than state space dimensions)
//   void* h             - pointer to value data
//   char* comment       - string to append to end of file
//
// Return Value:
//   int status          - zero if no error, negative if error
//
// Example Usage:
//
//   Assume that the value array for h has already been created with
//
//     int N[] = { 0, 25, 25, 25 }; // N[0] is not used
//     double*** h = (double***) makeValueArray( 3, &N[1] );
//
//   After the value data has been generated it can written to a file with
//
//     writeValueFile( "value.h", 3, 1, &N[1], h, "DP Value data" );
//
//   If no comment is desired, then NULL can be passed as the last parameter.
// ---------------------------------------------------------------------------
//
{
    ofstream fout( fname.data() );

    // Make sure we opened the file...

    if ( ! fout.is_open() ) return QNET_BAD_FILE;

    // Write header to file

    writeValueHeader( fout, ndim, nval, dim );

    // Compute array dimensions according to C/C++ convention; array
    // subscripts go from 0 to arrayDim[i]-1.

    int arrayNDim = ndim + ( nval > 1 ? 1 : 0 );
    int arrayDim[arrayNDim];
    for ( int i = 0; i < ndim; i++ )
    {
	arrayDim[i] = dim[i] + 1;
    }
    if ( nval > 1 ) arrayDim[ndim] = nval;

    // Write the value data to the file.

    writeValueData1D( fout, arrayNDim, arrayDim, (double*) h );

    // Write out comment

    writeComment( fout, comment );

    // Done writing, close the file.

    fout.close();

    return QNET_NORMAL;
}

//----------------------------------------------------------------------------

static void readValueData( ifstream& fin, int ndim, int dim[], double* h )
//
// ---------------------------------------------------------------------------
// This is a RECURSIVE function.
//
// Reads the data portion of a value file.  The arrays to contain the
// value data is assumed to have been created using the dynArray routines.
// If g is passed as NULL pointer then we do not want its data.
//
// Parameters:
//   ifstream& fin       - file stream to read data from
//   int ndim            - dimension of the arrays being read
//   int dim[ndim]       - array of largest indices in state space
//                          (one less than state space dimensions)
//   int* h              - pointer portion of h array being read
//
// Return Value:
//   none
// ---------------------------------------------------------------------------
//
{
    if ( ndim == 1 )
    {
	for ( int i = 0; i < dim[0]; i++ )
	{
	    fin.read( reinterpret_cast<char*> (&h[i]), sizeof( double ) );
	}
    }
    else if ( ndim > 1 )
    {
	ndim--;
	for ( int i = 0; i < dim[0]; i++ )
	{
	    readValueData( fin, ndim, &dim[1], ((double**)h)[i] );
	}
    }
}

//----------------------------------------------------------------------------

static void writeValueData( ofstream& fout, int ndim, int dim[], double* h )
//
// ---------------------------------------------------------------------------
// This is a RECURSIVE function.
//
// Writes the data portion of a value file.  The array containing the
// value data is assumed to have been created using the dynArray routines.
//
// Parameters:
//   ofstream& fout      - file stream to write data to
//   int ndim            - dimension of the array being written
//   int dim[ndim]       - array of largest indices in state space
//                          (one less than state space dimensions)
//   int* h              - pointer portion of h array being written
//
// Return Value:
//   none
// ---------------------------------------------------------------------------
//
{
    if ( ndim == 1 )
    {
	for ( int i = 0; i < dim[0]; i++ )
	{
	    fout.write( reinterpret_cast<char*> (&h[i]), sizeof( double ) );
	}
    }
    else if ( ndim > 1 )
    {
	ndim--;
	for ( int i = 0; i < dim[0]; i++ )
	{
	    writeValueData( fout, ndim, &dim[1], ((double**)h)[i] );
	}
    }
}

//----------------------------------------------------------------------------

int readValueFile( string fname, int ndim, int nval, int dim[], void* h )
//
// ---------------------------------------------------------------------------
// Reads a value data file into value arrays created with the dynArray
// routines.
//
// Parameters:
//   string fname        - name of file to read from
//   int ndim            - dimension of the array being read
//   int nval            - number of values per state space location
//   int dim[ndim]       - array of largest indices in state space
//                          (one less than state space dimensions)
//   void* h             - pointer to value data
//
// Return Value:
//   int status          - zero if no error, negative if error
//
// Example Usage:
//
//   To read a value file the dimension information must already be known
//   and the array or arrays to hold the value data must already be
//   declared.  The information necessary to do this can be obtained with
//   the getValueFileInfo() function.
//
//     int maxdim = 3; // largest possible dimension of data set state space
//     int ndim;
//     int nval;
//     int N[maxdim + 1];
//     getValueFileInfo( string("value.h"), maxdim, &ndim, &nval, &N[1] );
//
//   Convention used in most of the QNET DP programs says that the truncation
//   values are indexed starting with 1.  We can follow this convention by
//   declaring N[] to have one extra element and passing getValueFileInfo()
//   the address of the N[1] element.
//
//   If ndim = 3 and nval = 1 then we can allocate 3-dimensional array for h
//   with
//
//      double*** h = (double***) makeValueArray( ndim, nval, &N[1] );
//
//   Note that the only change necessary to the above line when ndim is 2
//   would be to use "**" in place of "***".  If, however, nval > 1 then
//   we also need an additional "dimension" to store the multiple values for
//   each state space location.  So, if ndim = 3 and nval = 2 the array h
//   should be created with
//
//      double**** h = (double****) makeValueArray( ndim, nval, &N[1] );
//
//   Finally, we're ready to read the data:
//
//      readValueFile( string("value.h"), ndim, nval, &N[1], h );
// ---------------------------------------------------------------------------
//
{
    int ndimInFile;        // Number of state space dimensions in file
    int nvalInFile;        // Number of values per state in file
    int dimInFile[ndim];   // State space dimensions in file
    int status;

    // Open file for reading.

    ifstream fin( fname.data() );

    // Read the value file header.

    if ( fin.is_open() )
    {
	status = readValueHeader( fin, ndim, &ndimInFile, &nvalInFile,
				  dimInFile );
    }
    else
    {
	status = QNET_BAD_FILE;
    }
    if ( status < 0 )
    {
	fin.close();
	return status;
    }

    // Check make sure we are expecting the correct number of dimensions
    // and values

    if ( ndimInFile != ndim )
    {
	fin.close();
	return QNET_BAD_DIMENSION;
    }

    if ( nvalInFile != nval )
    {
	fin.close();
	return QNET_BAD_NVALUES;
    }

    // If there is more than one value per state space location then we'll
    // need to increase the dimension of the array we read data into by one.

    int arrayNDim = ndim + ( nval > 1 ? 1 : 0 );
    int arrayDim[arrayNDim];
    for ( int i = 0; i < ndim; i++ )
    {
	if ( dim[i] != dimInFile[i] )
	{
	    fin.close();
	    return QNET_BAD_TRUNCATION;
	}
	arrayDim[i] = dim[i] + 1;
    }
    if ( nval > 1 ) arrayDim[ndim] = nval;

    // Read in the value data.

    readValueData( fin, arrayNDim, arrayDim, (double*) h );

    // All done

    fin.close();

    return QNET_NORMAL;
}

//----------------------------------------------------------------------------

int writeValueFile( string fname, int ndim, int nval, int dim[], void* h,
		    char* comment )
//
// ---------------------------------------------------------------------------
// Writes a value data file from value data stored in the array pointed
// to by p that was created with the dynArray routines.
//
// Parameters:
//   string fname        - name of file to write
//   int ndim            - dimension of the array being written
//   int nval            - number of values per state space location
//   int dim[ndim]       - array of largest indices in state space
//                          (one less than state space dimensions)
//   void* h             - pointer to value data
//   char* comment       - string to append to end of file
//
// Return Value:
//   int status          - zero if no error, negative if error
//
// Example Usage:
//
//   Assume that the value array for h has already been created with
//
//     int N[] = { 0, 25, 25, 25 }; // N[0] is not used
//     double*** h = (double***) makeValueArray( 3, &N[1] );
//
//   After the value data has been generated it can written to a file with
//
//     writeValueFile( "value.h", 3, 1, &N[1], h, "DP Value data" );
//
//   If no comment is desired, then NULL can be passed as the last parameter.
// ---------------------------------------------------------------------------
//
{
    ofstream fout( fname.data() );

    // Make sure we opened the file...

    if ( ! fout.is_open() ) return QNET_BAD_FILE;

    // Write header to file

    writeValueHeader( fout, ndim, nval, dim );

    // Compute array dimensions according to C/C++ convention; array
    // subscripts go from 0 to arrayDim[i]-1.

    int arrayNDim = ndim + ( nval > 1 ? 1 : 0 );
    int arrayDim[arrayNDim];
    for ( int i = 0; i < ndim; i++ )
    {
	arrayDim[i] = dim[i] + 1;
    }
    if ( nval > 1 ) arrayDim[ndim] = nval;

    // Write the value data to the file.

    writeValueData( fout, arrayNDim, arrayDim, (double*) h );

    // Write out comment

    writeComment( fout, comment );

    // Done writing, close the file.

    fout.close();

    return QNET_NORMAL;
}

//----------------------------------------------------------------------------

// End of file

