/*
 * $Id: pslice.cc,v 2.3 2009/10/17 02:16:51 senning Exp $
 *
 * Copyright (c) 2009 Department of Mathematics and Computer Science
 * Gordon College, 255 Grapevine Road, Wenham, MA 01984
 *
 * Author: Jonathan Senning <jonathan.senning@gordon.edu>
 * Written: October 16, 2009
 *
 * Displays a slice of policy data generated by a qnet program.
 *
 * Maintained by Jonathan Senning <jonathan.senning@gordon.edu>
 */

#include <iostream>
#include <fstream>
#include <sstream>
#include <string>
#include <cstdlib>
#include <math.h>
#include <qnet.h>
#include <CommandOptions.h>

//----------------------------------------------------------------------------
// Constants
//----------------------------------------------------------------------------

const char* Version = "2.0";
const char* Revision = "$Revision: 2.3 $";
const char* Copyright = "Copyright (c) 2009\n\
Department of Mathematics and Computer Science\n\
Gordon College, 255 Grapevine Road, Wenham MA, 01984";

const int MaxDim = 20;      // max policy dimension we can handle

//----------------------------------------------------------------------------
// Functions
//----------------------------------------------------------------------------

// Display short usage statement
//
// Input:
//   char* name    - character array containing program name

void usage( char *name )
{
    using namespace std;
    cout << "usage: " << name << " [OPTIONS] POLICY" << endl;
    cout << "Try " << name << " --help for more information" << endl;
}

//-------------------------------------------------------------------------- 

// Display general usage information
//
// Input:
//   char* name              - character array containing program name
//   CommandOptions& options - command line options object

void help( char* name, CommandOptions& options )
{
    using namespace std;
    cout << "usage: " << name << " [OPTIONS] POLICY" << endl;
    cout << endl;
    cout << "Displays policy data slice." << endl;
    cout << endl;
    cout << "OPTIONS may be one or more of" << endl;
    options.showOptionsHelp( 30 );
    cout << endl;
    cout << "SPEC is a slice specification.  A state is fully specified by "
	 << "x1,x2,...,xn"
	 << endl
	 << "if there are n classes in the network.  All but two of these "
	 << "values must be"
	 << endl
	 << "supplied to describe a two-dimension slice of the state space.  "
	 << "Assuming n=4,"
	 << endl
	 << "examples of valid slice specifications are"
	 << endl
	 << endl
	 << "    x1:3,x3:2   (x1=3, x2=free, x3=2, x4=free)" << endl
	 << "    x1:0,x2:0   (x1=0, x2=0, x3=free, x4=free)" << endl
	 << "    x3:5,x4:9   (x1=free, x2=free, x3=5, x4=9)" << endl
	 << endl
	 << "Whitespace is not allowed in the specification but '=' may be "
	 << "used in place"
	 << endl
	 << "of ':'."
	 << endl;
}

//----------------------------------------------------------------------------

// Convert a generic type to a string
// (From http://notfaq.wordpress.com/2006/08/30/c-convert-int-to-string/)
//
// Input:
//   const T& t    - generic value to convert to string
//
// Output:
//   std::string   - string version of input

template <class T>
inline std::string toString( const T& t )
{
    std::stringstream ss;
    ss << t;
    return ss.str();
}

//----------------------------------------------------------------------------

// Parse slice specification option string into array of state values.
//
// Input:
//   std::string str  - option argument
//   int x[]          - array of state values to fill
//   int n            - number of elements in x (starting with x[1])
//
// Output:
//   int x[]          - each entry (except x[0]) of is set
//   int              - number of state values found in option argument string
//
// Each entry of the array x[] is filled with either
// - the nonnegative value specified in the option argument string or
// - negative 1, indicating that the value was not specified
//
// Exactly two of the values of x[] must be set to -1, indicating the
// coordinates of the policy slice.

int parseBounds( std::string str, int x[], int n )
{
    using namespace std;

    const char* s = str.data();
    int length = str.length();

    int count = 0;
    int i = 0;

    for ( int i = 0; i <= n; i++ )
    {
	x[i] = -1;
    }

    while ( i < length )
    {
	if ( s[i] == 'x' || s[i] == 'X' )
	{
	    i++;
	    int classNum = 0;
	    bool atLeastOneDigit = false;
	    while ( i < length && s[i] >= '0' && s[i] <= '9' )
	    {
		classNum = 10 * classNum + ( s[i++] - '0' );
		atLeastOneDigit = true;
	    }
	    if ( classNum == 0 )
	    {
		cerr << "class numbers must be greater than zero" << endl;
		exit( EXIT_FAILURE );
	    }
	    if ( !atLeastOneDigit || ( s[i] != '=' && s[i] != ':' ) )
	    {
		cerr << "incorrectly formed slice specification" << endl;
		exit( EXIT_FAILURE );
	    }
	    i++;
	    int value = 0;
	    atLeastOneDigit = false;
	    while ( i < length && s[i] >= '0' && s[i] <= '9' )
	    {
		value = 10 * value + ( s[i++] - '0' );
		atLeastOneDigit = true;
	    }
	    if ( !atLeastOneDigit )
	    {
		cerr << "incorrectly formed slice specification" << endl;
		exit( EXIT_FAILURE );
	    }
	    if ( classNum > n )
	    {
		cerr << "class " << classNum << " is too big" << endl;
		exit( EXIT_FAILURE );
	    }		
	    x[classNum] = value;
	    count++;
	    if ( s[i] != '\0' ) i++;
	}
	else
	{
	    cerr << "incorrectly formed slice specification" << endl;
	    exit( EXIT_FAILURE );
	}
    }
    if ( count + 2 != n )
    {
	cerr << "exactly " << n - 2 << " state values must be specified"
	     << endl;
	exit( EXIT_FAILURE );
    }
    return count;
}

//----------------------------------------------------------------------------
//----------------------------------------------------------------------------

int main( int argc, char* argv[] )
{
    using std::cout;
    using std::cerr;
    using std::endl;
    using std::string;
    using std::vector;

    char *programName = argv[0];     // program name

    bool writeCSVFile = false;       // true to write policy slice to CSV file
    bool wantSlice = false;          // true if 2D slice is desired
    bool narrow = false;             // true to suppress spaces in slice output
    bool quiet = false;              // true to suppress nearly all output
    bool verbose = false;            // true to echo ALP solution parameters

    int maxTruncation = 0;
    int singleServer = 0;            // server number to display (0 = all)
    int status;

    string alpsFileName;             // name of ALP solution file (input)
    string policyFileName;           // name of full policy file
    string csvFileName;              // name of CSV policy slice file
    string boundsString;             // slice specification

    // Process the command line.
    // First we extract any options that may be present

    enum { ShowHelp, ShowVersion };
    CommandOptions options;
    options.setOption( 'c', "csv-file", "write CSV policy data to FILE",
		       CommandOptions::Required, "FILE" );
    options.setOption( 'N', "truncation",
		       "use N as the maximum truncation for each class",
		       CommandOptions::Required, "N" );
    options.setOption( 'n', "narrow", "do not insert spaces between actions" );
    options.setOption( 's', "server", "show policy information for server N",
		       CommandOptions::Required, "N" );
    options.setOption( 'S', "slice", "specify desired policy slice",
		       CommandOptions::Required, "SPEC" );
    options.setOption( 'q', "quiet", "suppress output" );
    options.setOption( 'v', "verbose", "show ALPS data" );
    options.setOption( ShowVersion, "version", "display program version" );
    options.setOption( ShowHelp, "help", "display usage information" );
    options.done();

    // Process all command line options

    int ch;
    while ( ( ch = options.getOption( argc, argv ) ) >= 0 )
    {
	switch ( ch )
	{
	    case 'c':
		writeCSVFile = true;
		csvFileName = string( options.getOptionArgument() );
		break;
	    case 'N':
		maxTruncation = atoi( options.getOptionArgument() );
		break;
	    case 'n':
		narrow = true;
		break;
	    case 'S':
		wantSlice = true;
		if ( options.getOptionArgument() != NULL )
		{
		    boundsString = string( options.getOptionArgument() );
		}
		break;
	    case 's':
		singleServer = atoi( options.getOptionArgument() );
		break;
	    case 'q':
		quiet = true;
		break;
	    case 'v':
		verbose = true;
		break;
	    case ShowHelp:
		help( programName, options );
		exit( EXIT_SUCCESS );
	    case ShowVersion:
		qnetVersion( programName, Version, Copyright );
		exit( EXIT_SUCCESS );
	    case '?':
	    default:
		help( programName, options );
		exit( EXIT_FAILURE );
	}
    }

    // Now adjust the argument count and pointers to skip over the options
    // we've just processed.

    argc -= options.getOptionCount();
    argv += options.getOptionCount();

    if ( argc != 2 )
    {
	usage( programName );
	exit( EXIT_FAILURE );
    }

    // Read policy data

    int numberOfClasses;
    int numberOfIndividualServers;
    int trunc[MaxDim + 1];
    status = getPolicyFileInfo( argv[1], MaxDim, &numberOfClasses,
				&numberOfIndividualServers, &trunc[1] );
    if ( status < 0 )
    {
	qnetError( status, "Cannot read policy file" );
	exit( EXIT_FAILURE );
    }

    int* policy = makePolicyArray1D( numberOfClasses,
				     numberOfIndividualServers, &trunc[1] );
    status = readPolicyFile1D( argv[1], numberOfClasses,
			       numberOfIndividualServers, &trunc[1], policy );
    if ( status < 0 )
    {
	cerr << "Could not read policy from " << argv[1] << endl;
	exit( EXIT_FAILURE );
    }

    // Report what we found...

    if ( verbose > 0 )
    {
	cout << endl
	     << "---- Policy Data ------------------------------"
	     << endl << endl;
	cout << "Data from " << argv[1] << endl;
	cout << "Dimension = " << numberOfClasses 
	     << "; Decisions = " << numberOfIndividualServers << endl;
	cout << "State space truncations: ";
	for ( int i = 1; i <= numberOfClasses; i++ )
	{
	    cout << ' ' << trunc[i];
	}
	cout << endl;
    }

    // Compute size of each dimension in policy data and compute truncations
    // to use for displaying the slice

    int dim[numberOfClasses + 2];
    int N[numberOfClasses + 1];
    for ( int i = 1; i <= numberOfClasses; i++ )
    {
	dim[i] = trunc[i] + 1;
	N[i] = trunc[i];
	if ( maxTruncation > 0 && maxTruncation <= N[i] )
	{
	    N[i] = maxTruncation;
	}
    }
    dim[numberOfClasses + 1] = numberOfIndividualServers;

    // check to make sure we have a slice to display

    if ( numberOfClasses < 2 )
    {
	cerr << "Policy data is " << numberOfClasses << "-dimensional but "
	     << "must be at least 2-dimensional." << endl;
	exit( EXIT_FAILURE );
    }

    if ( numberOfClasses > 2 && !wantSlice )
    {
	cerr << "Policy data is " << numberOfClasses << "-dimensional."
	     << endl
	     << "The command must include -S or --slice."
	     << endl
	     << "Use " << programName << " --help for more information"
	     << endl;
	exit( EXIT_FAILURE );
    }

    if ( verbose )
    {
	cout << endl
	     << "---- Policy Slice -----------------------------"
	     << endl;
    }

    if ( singleServer < 0 || singleServer > numberOfIndividualServers )
    {
	cerr << endl << "Incorrectly specified server; "
	     << "showing data for all servers" << endl;
	singleServer = 0;
    }

    // get the slice specification that was given on command line

    int x[numberOfClasses + 1];
    int numberOfBounds = 0;
    int n1 = 1;
    int n2 = 2;
    if ( wantSlice )
    {
	numberOfBounds = parseBounds( boundsString, x, numberOfClasses );

	// set n1 and n2 to the indices of the two dimensions defining the
	// policy slice we want.  This is done by by looking for the two
	// negative entries in the array x.

	int n  = 1;
	n1 = 0;
	n2 = 0;

	while ( n <= numberOfClasses && x[n] >= 0 ) n++; // find neg entry
	if ( n <= numberOfClasses )
	{
	    n1 = n++;
	}

	while ( n <= numberOfClasses && x[n] >= 0 ) n++; // find neg entry
	if ( n <= numberOfClasses )
	{
	    n2 = n++;
	}
	else
	{
	    cerr << endl << "could not identify policy slice" << endl;
	    exit( EXIT_FAILURE );
	}
    }

    // extract slice from policy data

    int action[N[n1]+1][N[n2]+1][numberOfIndividualServers];
    for ( int j = N[n2]; j >= 0; j-- )
    {
	x[n2] = j;
	for ( int i = 0; i <= N[n1]; i++ )
	{
	    x[n1] = i;
	    int offset = x[1];
	    for ( int k = 2; k <= numberOfClasses; k++ )
	    {
		offset = offset * ( dim[k] ) + x[k];
	    }
	    offset *= numberOfIndividualServers;
	    for ( int k = 0; k < numberOfIndividualServers; k++ )
	    {
		action[i][j][k] = policy[offset + k];
	    }
	}
    }

    if ( writeCSVFile )
    {
	// use CSV (comma separated value) format for output and write
	// to indicated file

	if ( !quiet )
	{
	    cout << endl
		 << "Saving policy slice data to " << csvFileName << endl;
	}

	std::ofstream fout( csvFileName.data() );
	if ( !fout.is_open() )
	{
	    cerr << "unable to open " << csvFileName << " for output" << endl;
	    exit( EXIT_FAILURE );
	}
	    
	fout << "Server:" << endl;
	if ( singleServer > 0 )
	{
	    fout << singleServer << endl;
	}
	else
	{
	    fout << "all" << endl;
	}
	if ( numberOfBounds > 0 )
	{
	    fout << endl << "Fixed state values:" << endl;
	    for ( int i = 1; i <= numberOfClasses; i++ )
	    {
		if ( i != n1 && i != n2 )
		{
		    fout << "x(" << i << ")=" << x[i] << endl;
		}
	    }
	}
	fout << endl;

	for ( int j = N[n2]; j >= 0; j-- )
	{
	    fout << "," << j << ",";
	    for ( int i = 0; i <= N[n1]; i++ )
	    {
		if ( singleServer > 0 )
		{
		    fout << ',' << action[i][j][singleServer - 1];
		}
		else
		{
		    fout << ",(" << action[i][j][0];
		    for ( int k = 1; k < numberOfIndividualServers; k++ )
		    {
			fout << '|' << action[i][j][k];
		    }
		    fout << ")";
		}
	    }
	    fout << endl;
	}
	fout << ",x(" << n2 << ")" << endl;
	fout << ",,x(" << n1 << ")";
	for ( int i = 0; i <= N[n1]; i++ )
	{
	    fout << ',' << i;
	}
	fout << endl;
	fout.close();
    }

    // display policy slice information

    if ( !quiet )
    {
	cout << endl << "Server: ";
	if ( singleServer > 0 )
	{
	    cout << singleServer << endl;
	}
	else
	{
	    cout << "all" << endl;
	}
	cout << "Slice is x(" << n1 << ") (horizontal) vs. x(" << n2
	     << ") (vertical): indexing begins with 0" << endl;
	if ( numberOfBounds > 0 )
	{
	    cout << "Fixed state values:" << endl;
	    for ( int i = 1; i <= numberOfClasses; i++ )
	    {
		if ( i != n1 && i != n2 )
		{
		    cout << " x(" << i << ")=" << x[i] << endl;
		}
	    }
	}
	cout << endl;

	for ( int j = N[n2]; j >= 0; j-- )
	{
	    for ( int i = 0; i <= N[n1]; i++ )
	    {
		if ( singleServer > 0 )
		{
		    if ( !narrow ) cout << ' ';
		    cout << action[i][j][singleServer - 1];
		}
		else
		{
		    if ( !narrow ) cout << ' ';
		    cout << "(" << action[i][j][0];
		    for ( int k = 1; k < numberOfIndividualServers; k++ )
		    {
			cout << '|' << action[i][j][k];
		    }
		    cout << ")";
		}
	    }
	    cout << endl;
	}
    }

    // Clean up

    destroyPolicyArray1D( policy, numberOfClasses, numberOfIndividualServers,
			  &trunc[1] );

    return 0;
}
