/*
 * $Id: qnetdp.cc,v 1.9 2009/10/21 12:34:53 senning Exp $
 *
 * Copyright (c) 2009 Jonathan R. Senning <jonathan.senning@gordon.edu>
 * Department of Mathematics and Computer Science
 * Gordon College, 255 Grapevine Road, Wenham, MA 01984
 *
 * Serial implementation of general dynamic programming program for multiclass
 * queueing or stochastic processing network control problems with either
 * deterministic or probabilistic routing.
 *
 * This program was written by Jonathan Senning and makes use of library
 * routines written by Jonathan Senning, Christopher Pfohl, and Taylor Carr.
 *
 * Maintained by Jonathan Senning <jonathan.senning@gordon.edu>
 *
 */

#include <iostream>
#include <sstream>
#include <iomanip>
#include <string>
#include <cstdlib>
#include <math.h>

#include <qnet.h>
#include <CommandOptions.h>
#include <SPNetwork.h>
#include <MultiIndex.h>

//-----------------------------------------------------------------------------

const char* Version = "1.0";
const char* Revision = "$Revision: 1.9 $";
const char* Copyright = "Copyright (c) 2009\n\
Department of Mathematics and Computer Science\n\
Gordon College, 255 Grapevine Road, Wenham, MA 01984";

const float DefaultEpsilon = 0.001;
const int   DefaultMaxIter = 10000;

//-----------------------------------------------------------------------------

template <class T>
inline std::string toString( const T& t )
//
// Convert a generic type to a string
//
// Input
//   const T& t      - a variable of a type that has an operator<<() method
//
// Output
//   std::string     - string version of variable
//
// (From http://notfaq.wordpress.com/2006/08/30/c-convert-int-to-string/)
//
{
    std::stringstream ss;
    ss << t;
    return ss.str();
}

//-----------------------------------------------------------------------------

void help( char* name, CommandOptions& options )
//
// Display short help message and information about command line options
//
// Input
//   char* name               - program name
//   CommandOptions& options  - instance of command-line options structure
//
// Output
//   information to standard output stream
//
{
    using namespace std;
    cout << "usage: " << name << " INPUT" << endl << endl;
    cout << "Computes the average cost per hour for a multiclass queueing"
	 << " or stochastic"
	 << endl
	 << "processing network using a dynamic program."
	 << endl << endl;
    cout << "See SPNetwork(5) for information regarding the syntax of the"
	 << " input file."
	 << endl << endl;
    cout << "OPTIONS may be one or more of" << endl;
    options.showOptionsHelp( 40 );
}

//-----------------------------------------------------------------------------

void doGreedy( SPNetwork& network, MultiIndex& state, double* h, double* w )
//
// Calculate one-stage cost plus future cost using a greedy policy
//
// Input
//   SPNetwork& network  - instance of SPNetwork
//   MultiIndex& state   - instance of MultiIndex; enumerates state space
//   double* h           - differential cost data
//
// Output
//   double* w           - updated differential cost data: h_new
//
{
    const int numberOfClasses = network.getClasses();
    const double* c           = network.getHoldingCosts();
    const double* lambda      = network.getArrivalRates();
    const double bigLambda    = network.getTotalEventRate();
    int muLength = 0;
    int pLength  = 0;
    SPNetwork::MatrixEntry* mu = network.getServiceRates( muLength );
    SPNetwork::MatrixEntry* p  = network.getTransitionProbabilities( pLength );
    std::vector<int**> networkFeasibleAction = network.getFeasibleActions();
    const unsigned int numberOfActions = networkFeasibleAction.size();

    const int* end = state.getCounterEnd( 1 );
    const int len = end[numberOfClasses];
    int* xx = state.getCounterArray( 1 );

    // initialize dxarr[] to zero and compute bigLambda-sum{lambda[i]}

    int dxarr[numberOfClasses + 1];
    double lambdaDiff = bigLambda;
    for ( int i = 1; i <= numberOfClasses; i++ )
    {
	dxarr[i] = 0;
	lambdaDiff -= lambda[i];
    }

    // create arrays to hold feasibility and cost information for each action

    bool isStateFeasible[numberOfActions];
    double cost[numberOfActions];

    // outer-loop; over all but last state space dimension

    for ( int x0 = state.start(); state.inRange(); x0 = state.next( 1 ) )
    {
	// inner-loop over last state space dimension

	for ( int x = x0; x <= x0 + len; x++ )
	{
	    xx[numberOfClasses] = x - x0; // compute last state value

	    // initialize values for this state

	    double holdingAndArrivalCost = 0.0;
	    for ( int j = 1; j <= numberOfClasses; j++ )
	    {
		dxarr[j] = 1; // job enters class j
		const int dx = state.getDelta( &dxarr[1] );
		dxarr[j] = 0; // restore current state
		holdingAndArrivalCost += c[j] * xx[j] + lambda[j] * h[x + dx];
	    }

	    for ( unsigned int z = 0; z < numberOfActions; z++ )
	    {
		cost[z] = lambdaDiff * h[x];
		int** u = networkFeasibleAction[z];
		isStateFeasible[z] = network.isStateFeasible( u, xx );
	    }

	    // loop over all possible state-to-state transitions

	    for ( int m = 0; m < pLength; m++ )
	    {
		const int j = p[m].i;
		const int k = p[m].j;
		dxarr[j] = -1; // job leaves class j
		dxarr[k] =  1; // job enters class k
		const int dx = state.getDelta( &dxarr[1] );
		dxarr[j] = 0; // restore to current state
		dxarr[k] = 0; // restore to current state
		const double pdh = p[m].val * ( h[x + dx] - h[x] );

		// loop over all network feasible actions
		
		for ( unsigned int z = 0; z < numberOfActions; z++ )
		{
		    if ( isStateFeasible[z] )
		    {
			int** u = networkFeasibleAction[z];

			// compute cost for current action in current state

			double sum = 0.0;
			for ( int n = 0; n < muLength; n++ )
			{
			    if ( mu[n].j == j )
			    {
				int i = mu[n].i;
				sum += u[i][j] * mu[n].val;
			    }
			}
			cost[z] += pdh * sum;
		    }
		}
	    }

	    // find minimum cost

	    double optimalCost = cost[0];
	    for ( unsigned int z = 1; z < numberOfActions; z++ )
	    {
		if ( isStateFeasible[z] && cost[z] < optimalCost )
		{
		    optimalCost = cost[z];
		}
	    }

	    // compute h_new (called w)

	    w[x] = ( holdingAndArrivalCost + optimalCost ) / bigLambda;
	}
    }
}

//-----------------------------------------------------------------------------

void update( MultiIndex& state, double* h, double* w,
	     double& cmin, double& cmax )
//
// Compute bounds on change to differential cost and shift new h so h[0] = 0.
//
// Input
//   MultiIndex& state   - instance of MultiIndex; enumerates state space
//   double* h           - differential cost data
//   double* w           - new differential cost data
//
// Output
//   double* h           - updated differential cost data
//   double& cmin        - minimum change in differential cost
//   double& cmax        - maximum change in differential cost
//
{
    double temp;
    const int* end = state.getCounterEnd( 1 );
    const int n    = state.getCounterDim();
    const int len  = end[n];
    cmin = cmax = w[0] - h[0];

    for ( int x0 = state.start(); state.inRange(); x0 = state.next( 1 ) )
    {
	for ( int x = x0; x <= x0 + len; x++ )
	{
	    temp = w[x] - h[x];
	    if ( temp < cmin ) cmin = temp;
	    if ( temp > cmax ) cmax = temp;
	    h[x] = w[x] - w[0];
	}
    }
}

//-----------------------------------------------------------------------------

void calcAction( SPNetwork& network, MultiIndex& state, double* h,
		 int numberOfIndividualServers, int* action )
//
// Compute the action in each state using a greedy policy and the
// differential cost function stored in the h[] array.
//
// Input
//   SPNetwork& network  - instance of SPNetwork
//   MultiIndex& state   - instance of MultiIndex; enumerates state space
//   double* h           - differential cost data
//   int numberOfIndividualServers - total number of servers in all pools
//
// Output
//   int* action         - action in each state
//
// This routine mirrors the doGreedy() routine except that it only computes
// the cost information in order to determine the action in each state.
{
    const int numberOfClasses = network.getClasses();
    const int numberOfServers = network.getServers();
    const int* K              = network.getServerPoolSizes();
    int muLength = 0;
    int pLength  = 0;
    SPNetwork::MatrixEntry* mu = network.getServiceRates( muLength );
    SPNetwork::MatrixEntry* p  = network.getTransitionProbabilities( pLength );
    std::vector<int**> networkFeasibleAction = network.getFeasibleActions();
    const unsigned int numberOfActions = networkFeasibleAction.size();

    const int* end = state.getCounterEnd( 1 );
    const int len = end[numberOfClasses];
    int* xx = state.getCounterArray( 1 );

    // initialize dxarr[] to zero

    int dxarr[numberOfClasses + 1];
    for ( int i = 1; i <= numberOfClasses; i++ )
    {
	dxarr[i] = 0;
    }

    // create arrays to hold feasibility and cost information for each action

    bool isStateFeasible[numberOfActions];
    double cost[numberOfActions];

    // outer-loop; over all but last state space dimension

    for ( int x0 = state.start(); state.inRange(); x0 = state.next( 1 ) )
    {
	// inner-loop over last state space dimension

	for ( int x = x0; x <= x0 + len; x++ )
	{
	    xx[numberOfClasses] = x - x0; // compute last state value

	    // initialize values for this state

	    for ( unsigned int z = 0; z < numberOfActions; z++ )
	    {
		cost[z] = 0.0;
		int** u = networkFeasibleAction[z];
		isStateFeasible[z] = network.isStateFeasible( u, xx );
	    }

	    // loop over all possible state-to-state transitions

	    for ( int m = 0; m < pLength; m++ )
	    {
		const int j = p[m].i;
		const int k = p[m].j;
		dxarr[j] = -1; // job leaves class j
		dxarr[k] =  1; // job enters class k
		const int dx = state.getDelta( &dxarr[1] );
		dxarr[j] = 0; // restore to current state
		dxarr[k] = 0; // restore to current state
		const double pdh = p[m].val * ( h[x + dx] - h[x] );

		// loop over all network feasible actions
		
		for ( unsigned int z = 0; z < numberOfActions; z++ )
		{
		    if ( isStateFeasible[z] )
		    {
			int** u = networkFeasibleAction[z];

			// compute cost for current action in current state

			double sum = 0.0;
			for ( int n = 0; n < muLength; n++ )
			{
			    if ( mu[n].j == j )
			    {
				int i = mu[n].i;
				sum += u[i][j] * mu[n].val;
			    }
			}
			cost[z] += pdh * sum;
		    }
		}
	    }

	    // find minimum cost and optimal action

	    double optimalCost = cost[0];
	    unsigned int bestAction = 0;
	    for ( unsigned int z = 1; z < numberOfActions; z++ )
	    {
		if ( isStateFeasible[z] && cost[z] < optimalCost )
		{
		    optimalCost = cost[z];
		    bestAction = z;
		}
	    }

	    // record optimal action.  The action array has one additional
	    // dimension than the state space in order to hold actions for
	    // each individual server.  The variable m is initialized to
	    // the first location for server actions for the current state.

	    int** u = networkFeasibleAction[bestAction];
	    int m = x * numberOfIndividualServers;
	    for ( int i = 1; i <= numberOfServers; i++ )
	    {
		int n = 0;
		for ( int j = 1; j <= numberOfClasses; j++ )
		{
		    for ( int k = 1; k <= u[i][j]; k++ )
		    {
			action[m++] = j;
			n++;
		    }
		}
		while ( n++ < K[i] )
		{
		    action[m++] = 0;
		}
	    }
	}
    }
}

//-----------------------------------------------------------------------------
//-----------------------------------------------------------------------------

int main( int argc, char* argv[] )
{
    using std::cout;
    using std::cerr;
    using std::endl;
    using std::setprecision;
    using std::string;

    int verbosity = 0;
    string policyFileName;
    string valueFileName;

    bool quiet = false;
    bool savePolicy = false;
    bool saveValue = false;

    char* programName = argv[0];

    // Get copy of command line

    string commandLine = string( argv[0] );
    for ( int i = 1; i < argc; i++ )
    {
        commandLine += string( " " ) + string( argv[i] );
    }

    // Process command line

    enum { ShowHelp, ShowVersion };

    CommandOptions options;
    options.setOption( 'h', "save-differential-cost",
		       "save computed differential cost to FILE",
		       CommandOptions::Required, "FILE" );
    options.setOption( 'p', "save-policy", "save computed policy to FILE",
		       CommandOptions::Required, "FILE" );
    options.setOption( 'q', "quiet", "minimize output" );
    options.setOption( 'v', "verbose", "show additional information" );
    options.setOption( ShowVersion, "version", "display program version" );
    options.setOption( ShowHelp, "help", "display usage information" );

    int ch;
    while ( ( ch = options.getOption( argc, argv ) ) >= 0 )
    {
	switch ( ch )
	{
	    case 'h':
		saveValue = true;
		valueFileName = string( options.getOptionArgument() );
		break;
	    case 'q':
		quiet = true;
		break;
	    case 'p':
		savePolicy = true;
		policyFileName = string( options.getOptionArgument() );
		break;
	    case 'v':
		verbosity++;
		break;
	    case ShowHelp:
		help( programName, options );
		return EXIT_SUCCESS;
	    case ShowVersion:
		qnetVersion( programName, Version, Copyright );
		return EXIT_SUCCESS;
	    default:
		help( programName, options );
		exit( EXIT_FAILURE );
	}
    }

    argc -= options.getOptionCount();
    argv += options.getOptionCount();

    // Only name of input file should be left on command line

    char* inputFileName = argv[1];

    if ( argc != 2 )
    {
	cerr << endl << "Error: missing input file" << endl << endl;
	help( programName, options );
	exit( EXIT_FAILURE );
    }

    // Read in data from input file

    TaggedValues input( inputFileName );
    SPNetwork network ( input );

    if ( !quiet )
    {
	cout << endl << "---- Parameters -------------------------------"
	     << endl << endl;
	network.printNetwork();
    }

    const int numberOfClasses = network.getClasses();
    const int numberOfServers = network.getServers();
    const int* K              = network.getServerPoolSizes();
    const double bigLambda    = network.getTotalEventRate();

    int N[numberOfClasses + 1];
    for ( int i = 1; i <= numberOfClasses; i++ )
    {
	string Ni = string( "N(" ) + toString( i ) + string( ")" );
	N[i] = input.getParamValue( Ni, 0 );
	if ( N[i] <= 0 )
	{
	    cerr << endl 
		 << "Error: input file does not have positive truncation "
		 << "value for N[" << i << "]"
	     << endl << endl;
	    exit( EXIT_FAILURE );
	}
    }

    double epsilon = input.getParamValue( "epsilon", 0.0 );
    if ( epsilon == 0.0 )
    {
	if ( verbosity > 0 )
	{
	    cerr << "Warning: "
		 << "input file does not have positive value for epsilon:"
		 << " using " << DefaultEpsilon << endl;
	}
	epsilon = DefaultEpsilon;
    }

    int maxIter = input.getParamValue( "maxIter", 0 );
    if ( maxIter == 0 )
    {
	if ( verbosity > 0 )
	{
	    cerr << "Warning: "
		 << "input file does not have positive value for maxIter:"
		 << " using " << DefaultMaxIter << endl;
	}
	maxIter = DefaultMaxIter;
    }

    if ( !quiet )
    {
	cout << "Truncations: N =";
	for ( int i = 1; i <= numberOfClasses; i++ )
	{
	    cout << " " << N[i];
	}
	cout << endl;
	cout << "epsilon = " << epsilon << endl;
	cout << "maxIter = " << maxIter << endl;
    }

    // Declare index to enumerate entire state space and arrays to hold
    // differential cost data

    MultiIndex state( numberOfClasses, &N[1] );
    double* h = new double [state.getCounterExtent()];
    double* w = new double [state.getCounterExtent()];

    for ( int x = state.start(); state.inRange(); x = state.next() )
    {
	h[x] = w[x] = 0.0;
    }

    // Value iteration loop

    double delta = 2.0 * epsilon;
    double cmin = 0.0, cmax = 0.0;
    int iter = 0;

    cout.precision( 16 );

    if ( !quiet )
    {
	cout << endl << "---- Solving ----------------------------------"
	     << endl;
    }

    double t1 = qnetTimer();

    while ( delta > epsilon && iter < maxIter )
    {
	doGreedy( network, state, h, w );
	update( state, h, w, cmin, cmax );
	if ( verbosity > 0 )
	{
	    cout << std::fixed << "iter = " << iter
		 << "\tcmin = "  << cmin << "\tcmax = " << cmax << endl;
	}
	delta = fabs( cmax - cmin );
	iter++;
    }

    double t2 = qnetTimer();

    // Output results

    if ( iter >= maxIter )
    {
	cout << endl
	     << "**** MAXIMUM NUMBER OF ITERATIONS EXCEEDED ****"
	     << endl << endl;
    }
    double J = 0.5 * ( cmin + cmax );
    int nDigits = int( log10( J ) + 1 + floor( -log10( epsilon ) ) );
    cout << endl
	 << "Solution found in " << setprecision( 4 )
	 << t2 - t1 << " seconds" << endl << endl;
    cout << "Number of iterations: " << iter << endl;
    cout << "Average cost per stage: " << setprecision( nDigits )
	 << J << endl;
    cout << "Average cost per hour: " << bigLambda * J << endl << endl;

    // Compute and save the optimal policy, if requested

    if ( savePolicy )
    {
	cout << "Saving policy data to " << policyFileName << "." << endl;

	int numberOfIndividualServers = 0;
	int mapSize = 0;
	for ( int i = 1; i <= numberOfServers; i++ )
	{
	    mapSize += K[i];
	}
	int serverMap[mapSize + 1];
	for ( int i = 1; i <= numberOfServers; i++ )
	{
	    for ( int j = 1; j <= K[i]; j++ )
	    {
		serverMap[++numberOfIndividualServers] = i;
	    }
	}

	if ( numberOfIndividualServers > numberOfServers )
	{
	    cout << "    The policy file contains data for individual servers."
		 << endl
		 << "    The server to server-pool mapping is:" << endl;
	    cout << "        Server    Pool" << endl
		 << "        ------    ----" << endl;
	    for ( int i = 1; i <= numberOfIndividualServers; i++ )
	    {
		cout << "         " << std::setw( 4 ) << i << "      "
		     << std::setw( 2 ) << serverMap[i] << endl;
	    }
	}

	int* action = makePolicyArray1D( numberOfClasses,
					 numberOfIndividualServers, &N[1] );

	calcAction( network, state, h, numberOfIndividualServers, action );

	writePolicyFile1D( policyFileName, numberOfClasses,
			   numberOfIndividualServers, &N[1], action, NULL );

	destroyPolicyArray1D( action, numberOfClasses,
			      numberOfIndividualServers, &N[1] );
    }

    // Save the differential cost data, if requested

    if ( saveValue )
    {
	cout << "Saving differential cost data to " << valueFileName << "."
	     << endl;
	writeValueFile1D( valueFileName, numberOfClasses, 1, &N[1], h, NULL );
    }

    // All done, clean up

    delete [] h;
    delete [] w;

    return 0;
}
